package theater;

import theater_intern.IComponent;
import theater_intern.IStage;
import util.Utils;

/**
 * Die Klasse Point reprsentiert ein Pixel auf der Bhne. Sie implementiert das
 * Interface PixelArea, so dass mit dieser Klasse Kollisionen von Pixeln mit
 * anderen Gebieten der Bhne berprft werden knnen.
 * 
 * @author Dietrich Boles, Universitt Oldenburg, Germany
 * @version 1.0 (08.09.2009)
 * 
 */
public class Point implements PixelArea {

	/**
	 * x-Koordinate des Punktes
	 */
	protected int x;

	/**
	 * y-Koordinate des Punktes
	 */
	protected int y;

	/**
	 * Konstruktor zum Initialisieren eines Punktes mit seiner x- und
	 * y-Koordinate.
	 * 
	 * @param x
	 *            x-Koordinate des Punktes
	 * @param y
	 *            y-Koordinate des Punktes
	 */
	public Point(int x, int y) {
		this.x = x;
		this.y = y;
	}

	/**
	 * Copy-Konstruktor zum Initialisieren eines Punktes mit einem
	 * java.awt.Point-Objekt
	 * 
	 * @param p
	 *            ein Objekt der Klase java.awt.Point (darf nicht null sein)
	 */
	public Point(java.awt.Point p) {
		this(p.x, p.y);
	}

	/**
	 * berprft, ob der Punkt mit den als Parameter bergebenen Koordinaten
	 * gleich dem aufgerufenen Punkt ist.
	 * 
	 * @param x
	 *            x-Koordinate des Punktes
	 * @param y
	 *            y-Koordinate des Punktes
	 * @return genau dann true, wenn die beiden Punkte gleich sind
	 * 
	 * @see theater.PixelArea#contains(int, int)
	 */
	public boolean contains(int x, int y) {
		return this.x == x && this.y == y;
	}

	/**
	 * berprft, ob der aufgerufene Punkt innerhalb der als Parameter
	 * bergebenen PixelArea liegt.
	 * 
	 * @param area
	 *            die zu vergleichende PixelArea
	 * @return genau dann true, wenn der aufgerufene Punkt innerhalb der als
	 *         Parameter bergebenen PixelArea liegt
	 * 
	 * @see theater.PixelArea#isInside(theater.PixelArea)
	 */
	public boolean isInside(PixelArea area) {
		// aus Performancegruenden
		if (Utils.dynInstanceof(area, Component.class)) {
			java.awt.Rectangle compRect = IComponent.getIComp((Component) area)
					.getImageRectangle();
			return compRect.contains(this.toAWTPoint());
		} else if (Utils.dynInstanceof(area, theater.Rectangle.class)) {
			java.awt.Rectangle pixelRect = ((theater.Rectangle) area)
					.toAWTRectangle();
			return pixelRect.contains(this.toAWTPoint());
		} else if (Utils.dynInstanceof(area, theater.Point.class)) {
			java.awt.Point p = ((theater.Point) area).toAWTPoint();
			return this.x == p.x && this.y == p.y;
		} else if (Utils.dynInstanceof(area, Stage.class)) {
			IStage stage = IStage.getIStage((Stage) area);
			java.awt.Rectangle compRect = new java.awt.Rectangle(0, 0, stage
					.getNumberOfColumns()
					* stage.getCellSize(), stage.getNumberOfRows()
					* stage.getCellSize());
			return compRect.contains(this.toAWTPoint());
		} else {
			return area.contains(this.x, this.y);
		}
	}

	/**
	 * berprft, ob der aufgerufene Punkt die als Parameter bergebene
	 * PixelArea schneidet, d.h. innerhalb der PixelArea liegt.
	 * 
	 * @param area
	 *            die zu vergleichende PixelArea
	 * @return genau dann true, wenn der aufgerufene Punkt innerhalb der als
	 *         Parameter bergebenen PixelArea liegt
	 * 
	 * @see theater.PixelArea#intersects(theater.PixelArea)
	 */
	public boolean intersects(PixelArea area) {
		return this.isInside(area);
	}

	/**
	 * Liefert die x-Koordinate des Punktes auf der Bhne.
	 * 
	 * @return die x-Koordinate des Punktes auf der Bhne
	 */
	public int getX() {
		return this.x;
	}

	/**
	 * Liefert die y-Koordinate des Punktes auf der Bhne.
	 * 
	 * @return die y-Koordinate des Punktes auf der Bhne
	 */
	public int getY() {
		return this.y;
	}

	/**
	 * Wandelt den Punkt in ein Objekt der Klasse java.awt.Point um.
	 * 
	 * @return der Punkt als java.awt.Point-Objekt
	 */
	public java.awt.Point toAWTPoint() {
		return new java.awt.Point(this.x, this.y);
	}

}
