package theater;

import java.util.List;

import theater_intern.IStage;

/**
 * Die Gestaltung einer konkreten Bhne kann durch die Definition einer von der
 * Klasse Stage abgeleiteten Klasse erfolgen. Eine Bhne besteht dabei aus einem
 * rechteckigen Gebiet, das sich aus gleichfrmigen quadratischen Zellen
 * zusammensetzt. Die Gre der Bhne wird ber einen Konstruktor durch die
 * Anzahl an Spalten und Reihen sowie die Gre der Zellen in Pixeln festgelegt.
 * Hierdurch wird ein Koordinatensystem definiert, das zum Platzieren von
 * Komponenten, d.h. Akteuren und Requisiten, auf der Bhne dient. Das
 * Koordinatensystem ist nicht endlich, so dass sich Akteure und Requisiten auch
 * auerhalb der Bhne befinden knnen, also (zwischenzeitlich) nicht sichtbar
 * sind.
 * <p>
 * </p>
 * Neben einer Menge von Getter-Methoden zum Abfragen des Zustands einer Bhne
 * sowie Methoden zur Verwaltung von Maus- und Tastatur-Events lassen sich die
 * Methoden der Klasse Stage einteilen in Methoden zur Gestaltung der Bhne und
 * Methoden zur Kollisionserkennung.
 * <p>
 * </p>
 * Zu den Gestaltungsmethoden gehren add- und remove-Methoden zum Platzieren
 * und Entfernen von Komponenten auf bzw. von der Bhne. Weiterhin existieren
 * Methoden zum Festlegen eines Hintergrundbildes fr die Bhne.
 * <p>
 * </p>
 * ber die Kollisionserkennungsmethoden lsst sich zur Laufzeit u. a.
 * ermitteln, welche Komponenten sich aktuell in bestimmten Bereichen der Bhne
 * aufhalten oder welche Komponenten (genauer gesagt deren Icons) sich berhren
 * oder berlappen. Die Klasse Stage implementiert das Interface PixelArea und
 * kann damit unmittelbar selbst in die Kollisionserkennung mit einbezogen
 * werden.
 * 
 * @author Dietrich Boles, Universitt Oldenburg, Germany
 * @version 1.0 (08.09.2009)
 * 
 */
public class Stage implements PixelArea {

	/**
	 * intern verwendets Objekt
	 */
	private IStage stage;

	/**
	 * ber den Konstruktor wird die Gre der Bhne festgelegt. Eine Bhne
	 * besteht dabei aus einem rechteckigen Gebiet, das sich aus gleichfrmigen
	 * quadratischen Zellen zusammensetzt. Die Gre der Bhne kann nachtrglich
	 * nicht mehr gendert werden.
	 * 
	 * @param noOfCols
	 *            Anzahl der Spalten der Bhne
	 * @param noOfRows
	 *            Anzahl der Reihen der Bhne
	 * @param cellSize
	 *            Gre einer Zelle in Pixeln
	 */
	public Stage(int noOfCols, int noOfRows, int cellSize) {
		super();
		this.stage = new IStage(noOfCols, noOfRows, cellSize);
		IStage.newStageIStage(this, this.stage);
	}

	/**
	 * Muss berschrieben werden und die Bhnen-spezifischen Aktionen enthalten
	 */
	public void run() {
	}

	/**
	 * Liefert die Anzahl an Spalten der Bhne.
	 * 
	 * @return die Anzahl an Spalten der Bhne
	 */
	public int getNumberOfColumns() {
		return this.stage.getNumberOfColumns();
	}

	/**
	 * Liefert die Anzahl an Reihen der Bhne.
	 * 
	 * @return die Anzahl an Reihen der Bhne
	 */
	public int getNumberOfRows() {
		return this.stage.getNumberOfRows();
	}

	/**
	 * Liefert die Gre der Zellen der Bhne in Pixeln
	 * 
	 * @return die Gre der Zellen der Bhne in Pixeln
	 */
	public int getCellSize() {
		return this.stage.getCellSize();
	}

	/**
	 * ndert die Gre der Bhne
	 * 
	 * @param noOfCols
	 *            Anzahl an Spalten
	 * @param noOfRows
	 *            Anzahl an Reihen
	 */
	public void setSize(int noOfCols, int noOfRows) {
		this.stage.setSize(noOfCols, noOfRows);
	}

	/**
	 * ndert die Gre der Bhne
	 * 
	 * @param noOfCols
	 *            Anzahl an Spalten
	 * @param noOfRows
	 *            Anzahl an Reihen
	 * @param cellSize
	 *            Gre jeder Zelle
	 */
	public void setSize(int noOfCols, int noOfRows, int cellSize) {
		this.stage.setSize(noOfCols, noOfRows, cellSize);
	}

	/**
	 * Platziert eine neue Komponente auf der Bhne. Angegeben werden die Spalte
	 * und Reihe der Zelle, auf der die Komponente platziert werden soll. Unter
	 * Platzierung ist dabei zu verstehen, dass der Mittelpunkt des Icons, das
	 * die Komponente reprsentiert, auf den Mittelpunkt der Zelle gesetzt wird.
	 * Da das Koordinatensystem der Bhne nicht begrenzt ist, kann eine
	 * Komponente auch auerhalb der Bhne platziert werden. Wenn die Komponente
	 * bereits auf der Bhne platziert ist, passiert nichts.
	 * <p>
	 * </p>
	 * Seiteneffekt: Nach erfolgreicher Platzierung der Komponente auf der Bhne
	 * wird die Methode addedToStage der Komponente mit dem entsprechenden
	 * Stage-Objekt als Parameter aufgerufen.
	 * <p>
	 * </p Achtung: Jede Komponente darf maximal einer Bhne zugeordnet sein.
	 * Ist bei Aufruf der Methode add die Komponente bereits einer anderen Bhne
	 * zugeordnet, muss sie bei dieser Bhne zunchst mittels remove entfernt
	 * werden. Ansonsten bleibt der Aufruf von add wirkungslos.
	 * 
	 * @param comp
	 *            die Komponente, die auf der Bhne platziert werden soll (darf
	 *            nicht null sein)
	 * @param col
	 *            die Spalte, in der die Komponente platziert werden soll
	 * @param row
	 *            die Reihe, in der die Komponente platziert werden soll
	 */
	public void add(Component comp, int col, int row) {
		this.stage.add(comp.comp, col, row);
	}

	/**
	 * Entfernt eine Komponente von der Bhne. Wenn die Komponente nicht auf der
	 * Bhne platziert ist, passiert nichts.
	 * 
	 * @param comp
	 *            die Komponente, die von der Bhne entfernt werden soll
	 */
	public void remove(Component comp) {
		this.stage.remove(comp.comp);
	}

	/**
	 * Entfernt alle Komponenten der Liste von der Bhne.
	 * 
	 * @param components
	 *            enthlt die Komponenten, die von der Bhne entfernt werden
	 *            sollen
	 */
	public void remove(List<Component> components) {
		this.stage.remove(components);
	}

	/**
	 * Ordnet der Bhne ein Hintergrundbild zu. Erlaubt sind Bilder der Formate
	 * gif, jpg und png. Das Bild wird mit der linken oberen Ecke auf die linke
	 * obere Ecke der Bhne platziert. Ist das Bild grer als die Bhne, wird
	 * rechts und/oder unten abgeschnitten. Ist das Bild kleiner als die Bhne,
	 * wird es repliziert dargestellt, bis die komplette Bhne berdeckt ist.
	 * 
	 * @param filename
	 *            Name der Bilddatei; die Datei muss sich im Unterverzeichnis
	 *            image des Theaterstck-Verzeichnisses befinden
	 * @throws IllegalArgumentException
	 *             wird geworfen, wenn die angegebene Datei keine gltige
	 *             lesbare Bilddatei ist
	 */
	public final void setBackground(String filename)
			throws IllegalArgumentException {
		this.stage.setBackground(filename);
	}

	/**
	 * Ordnet der Bhne ein TheaterImage als Hintergrundbild zu. Das Bild wird
	 * mit der linken oberen Ecke auf die linke obere Ecke der Bhne platziert.
	 * Ist das Bild grer als die Bhne, wird rechts und/oder unten
	 * abgeschnitten. Ist das Bild kleiner als die Bhne, wird es repliziert
	 * dargestellt, bis die komplette Bhne berdeckt ist.
	 * 
	 * @param image
	 *            das TheaterImage, das als Hintergrundbild verwendet werden
	 *            soll
	 */
	public final void setBackground(TheaterImage image) {
		this.stage.setBackground(image);
	}

	/**
	 * Liefert das Hintergrundbild der Bhne alsTheaterImage-Objekt. Wurde kein
	 * Hintergrundbild gesetzt, wird null geliefert.
	 * 
	 * @return das Hintergrundbild der Bhne als TheaterImage-Objekt
	 */
	public TheaterImage getBackground() {
		return this.stage.getBackground();
	}

	/**
	 * Zeichnet die Bhne. Normalerweise ist ein Aufruf dieser Methode nicht
	 * notwendig, da die Bhne bei nderungen automatisch aktualisiert wird.
	 * Wenn allerdings mittels der Methode getAWTImage der Klasse TheaterImage
	 * ein java.awt.image.BufferedImage-Objekt erfragt und gendert wird, muss
	 * diese Methode aufgerufen werden, wenn die nderungen des Image
	 * unmittelbar sichtbar gemacht werden sollen.
	 */
	public void paint() {
		this.stage.paint();
	}

	/**
	 * Liefert eine Liste mit allen Komponenten bestimmter Klassen, die aktuell
	 * auf der Bhne platziert sind. Fehlt der Parameter, werden alle
	 * Bhnen-Komponenten geliefert.
	 * 
	 * @param classes
	 *            Menge von Klassen, die bei der Suche bercksichtigt werden
	 *            sollen
	 * @return Liste mit allen Bhnen-Komponenten bestimmter Klassen
	 */
	public List<Component> getComponents(Class<?>... classes) {
		return this.stage.getComponents(classes);
	}

	/**
	 * Liefert eine Liste mit allen Komponenten bestimmter Klassen, die aktuell
	 * auf einer bestimmten Zelle der Bhne platziert sind. Werden keine
	 * Klassenobjekte bergeben, so werden alle Bhnen-Komponenten auf der
	 * entsprechenden Zelle geliefert.
	 * 
	 * @param column
	 *            Spalte der Zelle
	 * @param row
	 *            Reihe der Zelle
	 * @param classes
	 *            Menge von Klassen, die bei der Suche bercksichtigt werden
	 *            sollen
	 * @return Liste mit allen Bhnen-Komponenten bestimmter Klassen auf der
	 *         angegebenen Zelle
	 */
	public List<Component> getComponentsAt(int column, int row,
			Class<?>... classes) {
		return this.stage.getComponentsAt(column, row, classes);
	}

	/**
	 * Liefert eine Liste mit allen Komponenten bestimmter Klassen, deren
	 * zugeordnetes Icon vollstndig innerhalb einer bestimmten PixelArea liegt.
	 * Werden keine Klassenobjekte bergeben, so werden alle Bhnen-Komponenten
	 * innerhalb der PixelArea geliefert.
	 * 
	 * @param area
	 *            das Gebiet, in dem die Komponenten liegen sollen (darf nicht
	 *            null sein)
	 * @param classes
	 *            Menge von Klassen, die bei der Suche bercksichtigt werden
	 *            sollen
	 * @return Liste mit allen Bhnen-Komponenten bestimmter Klassen innerhalb
	 *         der angegebenen PixelArea
	 */
	public List<Component> getComponentsInside(PixelArea area,
			Class<?>... classes) {
		return this.stage.getComponentsInside(area, classes);
	}

	/**
	 * Liefert eine Liste mit allen Komponenten bestimmter Klassen, deren
	 * zugeordnetes Icon eine bestimmte PixelArea berhrt oder schneidet. Werden
	 * keine Klassenobjekte bergeben, so werden alle Bhnen-Komponenten
	 * geliefert, die die PixelArea berhren oder schneiden.
	 * 
	 * @param area
	 *            das Gebiet, das die Komponenten berhren oder schneiden sollen
	 *            (darf nicht null sein)
	 * @param classes
	 *            Menge von Klassen, die bei der Suche bercksichtigt werden
	 *            sollen
	 * @return Liste mit allen Bhnen-Komponenten bestimmter Klassen, die die
	 *         angegebene PixelArea berhren oder schneiden
	 */
	public List<Component> getIntersectingComponents(PixelArea area,
			Class<?>... classes) {
		return this.stage.getIntersectingComponents(area, classes);
	}

	/**
	 * berprft, ob der angegebene Punkt mit den Koordinaten x und y innerhalb
	 * der Bhne liegt.
	 * 
	 * @param x
	 *            x-Koordinate des Punktes
	 * @param y
	 *            y-Koordinate des Punktes
	 * @return true, falls der angegebene Punkt innerhalb der Bhne liegt
	 * 
	 * @see theater.PixelArea#contains(int, int)
	 */
	public boolean contains(int x, int y) {
		return new Rectangle(0, 0, this.stage.getNumberOfColumns()
				* this.stage.getCellSize(), this.stage.getNumberOfRows()
				* this.stage.getCellSize()).contains(x, y);
	}

	/**
	 * berprft, ob die Bhne vollstndig innerhalb der angegebenen PixelArea
	 * liegt.
	 * 
	 * @param area
	 *            das Gebiet, das berprft werden soll (darf nicht null sein)
	 * @return true, falls die Bhne vollstndig innerhalb der angegebenen
	 *         PixelArea liegt
	 * 
	 * @see theater.PixelArea#isInside(theater.PixelArea)
	 */
	public boolean isInside(PixelArea area) {
		return new Rectangle(0, 0, this.stage.getNumberOfColumns()
				* this.stage.getCellSize(), this.stage.getNumberOfRows()
				* this.stage.getCellSize()).isInside(area);
	}

	/**
	 * berprft, ob die Bhne eine angegebene PixelArea berhrt oder schneidet.
	 * 
	 * @param area
	 *            das Gebiet, das berprft werden soll (darf nicht null sein)
	 * @return true, falls die Bhne die angegebenen PixelArea berhrt oder
	 *         schneidet
	 * 
	 * @see theater.PixelArea#intersects(theater.PixelArea)
	 */
	public boolean intersects(PixelArea area) {
		return new Rectangle(0, 0, this.stage.getNumberOfColumns()
				* this.stage.getCellSize(), this.stage.getNumberOfRows()
				* this.stage.getCellSize()).intersects(area);
	}

	/**
	 * Legt fest, ob die Bhne Tastatur-Ereignisse behandeln soll. Standardmig
	 * ist dies der Fall.
	 * 
	 * @param handlingKeyEvents
	 *            true, falls die Bhne Tastatur-Ereignisse behandeln soll;
	 *            false andernfalls.
	 */
	public void setHandlingKeyEvents(boolean handlingKeyEvents) {
		this.stage.setHandlingKeyEvents(handlingKeyEvents);
	}

	/**
	 * berprft, ob die Bhne Tastatur-Ereignisse behandelt. Standardmig ist
	 * dies der Fall.
	 * 
	 * @return true, falls die Bhne Tastatur-Ereignisse behandelt
	 */
	public boolean isHandlingKeyEvents() {
		return this.stage.isHandlingKeyEvents();
	}

	/**
	 * Wird aufgerufen, wenn whrend die Bhne den Focus besitzt ein
	 * keyTyped-Event eingetreten ist. Soll eine Bhne auf keyTyped-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu keyTyped-Events finden sich in der Klasse
	 * java.awt.event.KeyListener. bergeben wird der Methode ein
	 * KeyInfo-Objekt, ber das Details zum eingetretenen Event abgefragt werden
	 * knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingKeyEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf keyTyped-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void keyTyped(KeyInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn whrend die Bhne den Focus besitzt ein
	 * keyPressed-Event eingetreten ist. Soll eine Bhne auf keyPressed-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu keyPressed-Events finden sich in der Klasse
	 * java.awt.event.KeyListener. bergeben wird der Methode ein
	 * KeyInfo-Objekt, ber das Details zum eingetretenen Event abgefragt werden
	 * knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingKeyEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf keyPressed-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void keyPressed(KeyInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn whrend die Bhne den Focus besitzt ein
	 * keyReleased-Event eingetreten ist. Soll eine Bhne auf keyReleased-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu keyReleased-Events finden sich in der Klasse
	 * java.awt.event.KeyListener. bergeben wird der Methode ein
	 * KeyInfo-Objekt, ber das Details zum eingetretenen Event abgefragt werden
	 * knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingKeyEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf keyReleased-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void keyReleased(KeyInfo e) {
	}

	/**
	 * Legt fest, ob die Bhne Maus-Ereignisse behandeln soll. Standardmig ist
	 * dies der Fall.
	 * 
	 * @param handlingMouseEvents
	 *            true, falls die Bhne Maus-Ereignisse behandeln soll; false
	 *            andernfalls.
	 */
	public void setHandlingMouseEvents(boolean handlingMouseEvents) {
		this.stage.setHandlingMouseEvents(handlingMouseEvents);
	}

	/**
	 * berprft, ob die Bhne Maus-Ereignisse behandelt. Standardmig ist dies
	 * der Fall.
	 * 
	 * @return true, falls die Bhne Maus-Ereignisse behandelt
	 */
	public boolean isHandlingMouseEvents() {
		return this.stage.isHandlingMouseEvents();
	}

	/**
	 * Wird aufgerufen, wenn ein mousePressed-Event auf der Bhne eingetreten
	 * ist, d.h. eine Maustaste gedrckt wird, whrend sich der Mauszeiger
	 * oberhalb der Bhne befindet. Soll eine Bhne auf mousePressed-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu mousePressed-Events finden sich in der Klasse
	 * java.awt.event.MouseListener. bergeben wird der Methode ein
	 * MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mousePressed-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mousePressed(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseReleased-Event auf der Bhne eingetreten
	 * ist, d.h. eine gedrckte Maustaste losgelassen wird, whrend sich der
	 * Mauszeiger oberhalb der Bhne befindet. Soll eine Bhne auf
	 * mouseReleased-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu mouseReleased-Events finden sich
	 * in der Klasse java.awt.event.MouseListener. bergeben wird der Methode
	 * ein MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mouseReleased-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseReleased(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseClicked-Event auf der Bhne eingetreten
	 * ist, d.h. eine Maustaste geklickt wurde, whrend sich der Mauszeiger
	 * oberhalb der Bhne befindet. Soll eine Bhne auf mouseClicked-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu mouseClicked-Events finden sich in der Klasse
	 * java.awt.event.MouseListener. bergeben wird der Methode ein
	 * MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mouseClicked-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseClicked(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseDragged-Event auf der Bhne eingetreten
	 * ist, d.h. die Maus bei gedrckter Maustaste bewegt wurde, whrend sich
	 * der Mauszeiger oberhalb der Bhne befindet. Soll eine Bhne auf
	 * mouseDragged-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu mouseDragged-Events finden sich
	 * in der Klasse java.awt.event.MouseMotionListener. bergeben wird der
	 * Methode ein MouseInfo-Objekt, ber das Details zum eingetretenen Event
	 * abgefragt werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mouseDragged-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseDragged(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseMoved-Event auf der Bhne eingetreten ist,
	 * d.h. die Maus bewegt wurde, whrend sich der Mauszeiger oberhalb der
	 * Bhne befindet. Soll eine Bhne auf mouseMoved-Events reagieren, muss sie
	 * diese Methode entsprechend berschreiben. Genauere Informationen zu
	 * mouseMoved-Events finden sich in der Klasse
	 * java.awt.event.MouseMotionListener. bergeben wird der Methode ein
	 * MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mouseMoved-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseMoved(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseEntered-Event auf der Bhne eingetreten
	 * ist, d.h. der Mauszeiger auf die Bhne gezogen wird. Soll eine Bhne auf
	 * mouseEntered-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu mouseEntered-Events finden sich
	 * in der Klasse java.awt.event.MouseListener. bergeben wird der Methode
	 * ein MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mouseEntered-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseEntered(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseExited-Event auf der Bhne eingetreten
	 * ist, d.h. der Mauszeiger die Bhne verlsst. Soll eine Bhne auf
	 * mouseExited-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu mouseExited-Events finden sich
	 * in der Klasse java.awt.event.MouseListener. bergeben wird der Methode
	 * ein MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl die Bhne als auch Komponenten knnen auf mouseExited-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseExited(MouseInfo e) {
	}

}
