package theater_intern;

import java.util.HashMap;

import listener.Application;
import listener.StandAloneApplet;
import model.Play;
import simulation.SimulationManager;
import theater.Performance;

/**
 * Interne Umsetzung einer Performance
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class IPerformance {

	/***************************************************************************
	 * * * * * * * * * * * * * * Umformungsmethoden
	 */
	private static HashMap<Performance, IPerformance> perfMap = new HashMap<Performance, IPerformance>();
	private static HashMap<IPerformance, Performance> iPerfMap = new HashMap<IPerformance, Performance>();

	public static void newPerfIPerf(Performance perf, IPerformance iPerf) {
		IPerformance.perfMap.put(perf, iPerf);
		IPerformance.iPerfMap.put(iPerf, perf);
	}

	public static Performance getPerformance(IPerformance iPerf) {
		return IPerformance.iPerfMap.get(iPerf);
	}

	public static IPerformance getIPerformance(Performance perf) {
		return IPerformance.perfMap.get(perf);
	}

	private static IPerformance defaultPerformance = null;

	/***************************************************************************
	 * * * * * * * * * * * * * * Performancemethoden
	 */

	final public static int STOPPED = 0;
	final public static int RUNNING = 2;

	private int state;
	private int speed;
	private boolean freeze;

	SoundHandler soundHandler;
	
	public static void reinit() {
		defaultPerformance = null;
		perfMap = new HashMap<Performance, IPerformance>();
		iPerfMap = new HashMap<IPerformance, Performance>();
	}

	public static IPerformance getDefaultPerformance() {
		if (IPerformance.defaultPerformance == null) {
			Performance defPerformance = new Performance();
			IPerformance.defaultPerformance = IPerformance
					.getIPerformance(defPerformance);
		}
		return IPerformance.defaultPerformance;
	}

	public IPerformance() {
		this.state = IPerformance.STOPPED;
		this.speed = Performance.DEF_SPEED;
		this.freeze = false;
		if (Application.isApplet()) {
			this.soundHandler = new SoundAsClip(StandAloneApplet.applet);
		} else {
			this.soundHandler = new SoundAsStream();
		}
	}

	public void startSimulation() {
		if (this.state != IPerformance.STOPPED) {
			return;
		}
		IPerformance.getPerformance(this).started();
		this.state = IPerformance.RUNNING;
		TheaterObservable.getObservable().performanceStateChange();
	}

	public void stopSimulation() {
		if (this.state == IPerformance.STOPPED) {
			return;
		}
		SimulationManager.getSimulationManager().handleAPIStop();
		this.state = IPerformance.STOPPED;
		stopSounds();
		IPerformance.getPerformance(this).stopped();
		TheaterObservable.getObservable().performanceStateChange();
	}

	public void setStopped() {
		if (this.state == IPerformance.STOPPED) {
			return;
		}
		this.state = IPerformance.STOPPED;
		stopSounds();
		IPerformance.getPerformance(this).stopped();
		TheaterObservable.getObservable().performanceStateChange();
	}

	public void resetSimulation() {
		if (!this.simulationStopped()) {
			this.stopSimulation();
		}
		Play.getPlay().reset();
	}

	public boolean simulationRunning() {
		return this.state == IPerformance.RUNNING;
	}

	public boolean simulationStopped() {
		return this.state == IPerformance.STOPPED;
	}

	public void freeze() {
		this.freeze = true;
	}

	public void unfreeze() {
		this.freeze = false;
		TheaterObservable.getObservable().importantStateChange();
	}

	public boolean frozen() {
		return this.freeze;
	}

	public void playSound(String soundFile) {
		this.soundHandler.playSound(soundFile);

	}

	public void stopSounds() {
		this.soundHandler.stopSounds();
	}

	public int getSimulationSpeed() {
		return this.speed;
	}

	public void setSimulationSpeedAPI(int newSpeed) {
		if (newSpeed != this.speed) {
			this.speed = newSpeed;
			IPerformance.getPerformance(this).speedChanged(this.speed);
			if (Play.getPlay().getPlayFrame() != null) {
				Play.getPlay().getPlayFrame().setSpeed(newSpeed);
			}
		}
	}

	public void setSimulationSpeed(int newSpeed) {
		if (newSpeed != this.speed) {
			this.speed = newSpeed;
		}
	}
}
