package theater;

import model.Play;
import theater_intern.IPerformance;
import theater_intern.IStage;

/**
 * Die Klasse Performance definiert Methoden zur Steuerung und Verwaltung der
 * Ausfhrung von Theater-Programmen(stop, setSpeed, freeze, unfreeze).
 * <p>
 * </p>
 * Weiterhin werden Methoden definiert, die unmittelbar nach entsprechenden
 * Steuerungsaktionen aufgerufen werden, und zwar auch, wenn die Aktionen durch
 * die Steuerungsbuttons des Simulators ausgelst wurden(started, stopped,
 * speedChanged). Mchte ein Programmierer zustzliche Aktionen mit den
 * entsprechenden Steuerungsaktionen einhergehen lassen, kann er eine
 * Unterklasse der Klasse Performance definieren und hierin die entsprechende
 * Methode berschreiben.
 * <p>
 * </p>
 * Zustzlich stellt die Klasse Performance eine Methode playSound zur
 * Verfgung, mit der eine Audio-Datei abgespielt werden kann. Die Datei muss
 * sich im Unterverzeichnis "sounds" des entsprechenden Theraterstcks befinden.
 * Untersttzt werden die Formate wav, au und aiff.
 * <p>
 * </p>
 * ber die Methode setActiveStage ist es mglich, die aktuell aktive Bhne
 * gegen eine andere Bhne auszutauschen, d.h. das Bhnenbild zu wechseln. Unter
 * Umstnden aktive Akteure der alten Bhne werden dabei nicht automatisch
 * gestoppt. Die Methode getActiveStage liefert die gerade aktive Bhne.
 * <p>
 * </p>
 * Das whrend einer Auffhrung aktuelle Performance-Objekt kann mit Hilfe der
 * statischen Methode getPerformance ermittelt werden. Ein Wechsel des
 * Performance-Objektes ist whrend einer Auffhrung nicht mglich.
 * 
 * @author Dietrich Boles, Universitt Oldenburg, Germany
 * @version 1.0 (08.09.2009)
 * 
 */
public class Performance {

	/**
	 * Minimale Geschwindigkeit
	 */
	public static int MIN_SPEED = 0;

	/**
	 * Maximale Geschwindigkeit
	 */
	public static int MAX_SPEED = 100;

	/**
	 * Default-Geschwindigkeit
	 */
	public static int DEF_SPEED = 50;

	/**
	 * intern verwendetes Objekt
	 */
	private IPerformance perf;

	/**
	 * Default-Konstruktor zur Initialisierung eines Performance-Objektes. Der
	 * Zustand des Performance-Objektes wird auf STOPPED gesetzt, die
	 * Geschwindigkeit auf DEF_SPEED.
	 */
	public Performance() {
		this.perf = new IPerformance();
		IPerformance.newPerfIPerf(this, this.perf);
	}

	/**
	 * Liefert das aktuell gesetzte Performance-Objekte
	 * 
	 * @return das aktuell gesetzte Performance-Objekt
	 */
	public static Performance getPerformance() {
		return IPerformance.getPerformance(Play.getPlay()
				.getActivePerformance());
	}

	/**
	 * Stopped eine Performance und setzt sie in den Zustand STOPPED. Wenn sich
	 * die Performance bereits im Zustand STOPPED befindet, passiert nichts.
	 * <p>
	 * </p>
	 * Die Methode kann aus einem Programm heraus aufgerufen werden. Sie wird
	 * auch aufgerufen, wenn der Nutzer im Theater-Simulator den Stopp-Button
	 * anklickt.
	 */
	public void stop() {
		this.perf.stopSimulation();
	}

	/**
	 * Diese Methode wird unmittelbar nach dem Starten bzw. Fortsetzen eines
	 * Theaterstcks aufgerufen.
	 * <p>
	 * </p>
	 * Sollen mit dem Start eines Theaterstcks weitere Aktionen einhergehen,
	 * muss der Programmierer eine Klasse von der Klasse Performance ableiten
	 * und diese Methode entsprechend berschreiben.
	 */
	public void started() {
	}

	/**
	 * Diese Methode wird unmittelbar nach dem Stoppen eines Theaterstcks
	 * aufgerufen.
	 * <p>
	 * </p>
	 * Sollen mit dem Stoppen eines Theater-Programms weitere Aktionen
	 * einhergehen, muss der Programmierer eine Klasse von der Klasse
	 * Performance ableiten und diese Methode entsprechend berschreiben.
	 */
	public void stopped() {
	}

	/**
	 * Diese Methode wird aufgerufen, wenn sich die Geschwindigkeit des
	 * Theaterstcks ndert.
	 * <p>
	 * </p>
	 * Sollen mit einer Geschwindigkeitsnderung weitere Aktionen einhergehen,
	 * muss der Programmierer eine Klasse von der Klasse Performance ableiten
	 * und diese Methode entsprechend berschreiben.
	 * 
	 * @param newSpeed
	 *            die neue Geschwindigkeit
	 */
	public void speedChanged(int newSpeed) {
	}

	/**
	 * ndert die aktuell eingestellte Ausfhrungsgeschwindigkeit. Die minimale
	 * Geschwindigkeit kann ber die Konstante MIN_SPEED, die maximale
	 * Geschwindigkeit ber die Konstante MAX_SPEED abgefragt werden. Wird als
	 * gewnschte Geschwindigkeit ein kleinerer bzw. grerer Wert als die
	 * entsprechende Konstante bergeben, wird die Geschwindigkeit automatisch
	 * auf MIN_SPEED bzw. MAX-SPEED gesetzt.
	 * <p>
	 * </p>
	 * Die Methode kann aus einem Programm heraus aufgerufen werden. Sie wird
	 * auch aufgerufen, wenn der Nutzer im Theater-Simulator den
	 * Geschwindigkeitsregler benutzt.
	 * 
	 * @param newSpeed
	 *            die neue Ausfhrungsgeschwindigkeit, die zwischen MIN_SPEED
	 *            und MAX_SPEED liegen sollte
	 */
	public void setSpeed(int newSpeed) {
		if (newSpeed < Performance.MIN_SPEED) {
			newSpeed = Performance.MIN_SPEED;
		} else if (newSpeed > Performance.MAX_SPEED) {
			newSpeed = Performance.MAX_SPEED;
		}
		this.perf.setSimulationSpeedAPI(newSpeed);
	}

	/**
	 * Liefert die aktuelle Ausfhrungsgeschwindigkeit der Performance.
	 * 
	 * @return die aktuelle Ausfhrungsgeschwindigkeit der Performance
	 */
	public int getSpeed() {
		return this.perf.getSimulationSpeed();
	}

	/**
	 * Spielt einen Sound ab, der aus einer Datei geladen wird. Erlaubt sind die
	 * Formate au, aiff und wav.
	 * 
	 * @param soundFile
	 *            Name der Sounddatei; die Datei muss sich im Unterverzeichnis
	 *            sounds des Theaterstck-Verzeichnisses befinden
	 * @throws IllegalArgumentException
	 *             wird geworfen, wenn die angegebene Datei keine gltige
	 *             lesbare Sounddatei ist
	 */
	public void playSound(String soundFile) throws IllegalArgumentException {
		this.perf.playSound(soundFile);
	}

	/**
	 * Der Aufruf dieser Methode fhrt dazu, dass die Ansicht der Bhne
	 * "eingeforen" wird, d.h. es werden keinerlei Zustandsnderungen mehr
	 * sichtbar, bevor nicht die Methode unfreeze aufgerufen worden ist.
	 */
	public void freeze() {
		this.perf.freeze();
	}

	/**
	 * Bei Aufruf dieser Methode wird der Eingefroren-Zustand wieder verlassen.
	 * Befindet sich der aufrufene Thread nicht im Eingefroren-Zustand, bewirkt
	 * ein Aufruf dieser Methode nichts.
	 */
	public void unfreeze() {
		this.perf.unfreeze();
	}

	/**
	 * Wechselt die Bhne. Der Bhnenwechsel wird nicht unmittelbar beim Aufruf
	 * der Methode sondern erst nach Abschluss der run-Runde durchgefhrt.
	 * 
	 * @param stage
	 *            die neue Bhne
	 */
	public void setActiveStage(Stage stage) {
		IStage.setStage(IStage.getIStage(stage));
	}

	/**
	 * Liefert die aktuell dargestellte Bhne
	 * 
	 * @return die aktuell dargestellte Bhne
	 */
	public Stage getActiveStage() {
		return IStage.getStage(Play.getPlay().getActiveStage());
	}

}
