package model;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.ArrayList;

import javax.xml.namespace.QName;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import listener.Application;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import theater.Actor;
import theater.Performance;
import theater.Prop;
import theater.Stage;
import theater_intern.IPerformance;
import theater_intern.IStage;
import theater_intern.TheaterObservable;
import util.IO;
import util.ResourceLoader;
import view.PlayFrame;
import view.StagePanel;

import compiler.CompileManager;

import editor.Editor;
import editor.TextEditor;

/**
 * Verwaltet die internen Daten eines Theaterstcks. Observer werden informiert,
 * wenn sich die Bhne gendert hat.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class Play {

	public static final String CLASSES = "Classes";
	public static final String STAGE = "Stage";
	public static final String ACTOR = "Actor";
	public static final String PROP = "Prop";
	public static final String PERFORMANCE = "Performance";
	public static final String OTHER = "Other";

	private PlayFrame playFrame;
	private StagePanel stagePanel;
	private String directory;
	private ArrayList<String> performances;
	private ArrayList<String> stages;
	private ArrayList<String> actors;
	private ArrayList<String> props;
	private ArrayList<String> others;
	String activeStageName;
	private IStage activeStage;
	String activePerformanceName;
	private IPerformance activePerformance;

	private TextEditor infoEditor;

	private static Play play;
	private boolean isEventQueueActive;
	private Object syncObject = new Object();

	/**
	 * Liefert das aktuelle Theaterstck
	 * 
	 * @return
	 */
	public static Play getPlay() {
		return Play.play;
	}

	/**
	 * Erzeugt eines leeres Theaterstck (von dem aus dann ein neues erzeugt
	 * oder ein existierendes geladen werden kann)
	 */
	public Play() {
		Play.play = this;
		this.performances = new ArrayList<String>();
		this.stages = new ArrayList<String>();
		this.actors = new ArrayList<String>();
		this.props = new ArrayList<String>();
		this.others = new ArrayList<String>();
		this.activeStageName = null;
		this.activeStage = null;
		this.activePerformanceName = null;
		this.activePerformance = IPerformance.getDefaultPerformance();
		this.directory = null;
		this.playFrame = new PlayFrame(this);
		this.stagePanel = this.playFrame.getStagePanel();
		this.infoEditor = null;
		this.isEventQueueActive = false;
	}

	/**
	 * ffnet ein existierendes Theaterstck; bergeben wird die entsprechende
	 * Theater-XML-Datei
	 * 
	 * @param playDescFile
	 */
	public Play(File playDescFile) { // existing play
		Play.play = this;
		this.performances = new ArrayList<String>();
		this.stages = new ArrayList<String>();
		this.actors = new ArrayList<String>();
		this.props = new ArrayList<String>();
		this.others = new ArrayList<String>();
		this.directory = playDescFile.getParent();
		this.activeStageName = null;
		this.activeStage = null;
		this.activePerformanceName = null;
		this.activePerformance = IPerformance.getDefaultPerformance();

		// this.load(playDescFile);
		new SAXHandler(this).load(playDescFile);
		boolean success = CompileManager.getCompileManager().compile(this);

		try {
			if (success && this.activePerformanceName != null) {
				this.activePerformance = IPerformance
						.getIPerformance((Performance) CompileManager
								.getCompileManager().getClassLoader()
								.loadClass(this.activePerformanceName)
								.newInstance());
			}
		} catch (Throwable exc) {
			exc.printStackTrace();
		}

		try {
			if (success && this.activeStageName != null) {
				this.activeStage = IStage.getIStage((Stage) CompileManager
						.getCompileManager().getClassLoader().loadClass(
								this.activeStageName).newInstance());
			}
		} catch (Throwable exc) {
			exc.printStackTrace();
		}

		this.playFrame = new PlayFrame(this);
		this.stagePanel = this.playFrame.getStagePanel();
		this.infoEditor = new TextEditor(this.directory + File.separatorChar
				+ "Info.txt", this.getName());
	}

	/**
	 * ffnet ein existierendes Theaterstck; bergeben wird die entsprechende
	 * Theater-XML-Datei (fr Standalone)
	 * 
	 * @param playDescFile
	 */
	public Play(String playDescFile, boolean dummy) { // existing play
		Play.play = this;
		this.performances = new ArrayList<String>();
		this.stages = new ArrayList<String>();
		this.actors = new ArrayList<String>();
		this.props = new ArrayList<String>();
		this.others = new ArrayList<String>();
		this.directory = "data";
		this.activeStageName = null;
		this.activeStage = null;
		this.activePerformanceName = null;
		this.activePerformance = IPerformance.getDefaultPerformance();

		new SAXHandler(this).load(playDescFile);
		try {
			if (this.activePerformanceName != null) {
				this.activePerformance = IPerformance
						.getIPerformance((Performance) Class.forName(
								"data." + this.activePerformanceName)
								.newInstance());
			}
			this.activeStage = IStage.getIStage((Stage) Class.forName(
					"data." + this.activeStageName).newInstance());

			this.playFrame = new PlayFrame(this);
			this.stagePanel = this.playFrame.getStagePanel();
			this.infoEditor = new TextEditor(ResourceLoader
					.getDataFileInputStream("data/Info.txt"), this.getName());
		} catch (Throwable th) {
			th.printStackTrace();
		}
	}

	/**
	 * Erzeugt ein neues Theaterstck; bergeben wird der Name des
	 * Verzeichnisses
	 * 
	 * @param dirName
	 */
	public Play(String dirName) { // new play
		try {
			Play.play = this;
			this.performances = new ArrayList<String>();
			this.stages = new ArrayList<String>();
			this.actors = new ArrayList<String>();
			this.props = new ArrayList<String>();
			this.others = new ArrayList<String>();
			this.directory = dirName;
			this.activeStageName = null;
			this.activeStage = null;
			this.activePerformanceName = null;
			this.activePerformance = IPerformance.getDefaultPerformance();

			this.create(this.directory);

			this.playFrame = new PlayFrame(this);
			this.stagePanel = this.playFrame.getStagePanel();

			this.infoEditor = new TextEditor(this.directory
					+ File.separatorChar + "Info.txt", this.getName());
			this.save();
		} catch (Throwable exc) {
			exc.printStackTrace();
		}
	}

	/**
	 * Abfrage, ob es sich um ein leeres Theaterstck handelt. Ein Theaterstck
	 * ist leer, wenn es nur den Frame zum Erzeugen neuer oder ffnen
	 * existierender Theaterstcke bereitstellt.
	 * 
	 * @return
	 */
	public boolean isEmpty() {
		return this.activeStage == null && this.directory == null;
	}

	/**
	 * Liefert den Namen des Theaterstcks (entspricht dem Verzeichnisnamen)
	 * 
	 * @return
	 */
	public String getName() {
		if (this.isEmpty()) {
			return "";
		} else {
			return new File(this.directory).getName();
		}
	}

	/**
	 * Liefert den PlayFrame des Theaterstcks (ist anfangs null)
	 * 
	 * @return
	 */
	public PlayFrame getPlayFrame() {
		return this.playFrame;
	}

	/**
	 * Liefert den StagePanel des Theaterstcks (ist anfangs null)
	 * 
	 * @return
	 */
	public StagePanel getStagePanel() {
		return this.stagePanel;
	}

	/**
	 * setzt den Namen der aktiven Bhne und ldt das entsprechende Bhnenobjekt
	 * 
	 * @param name
	 */
	public void setActiveStageName(String name) {
		if (name == null && this.activeStageName == null) {
			return;
		}
		if (name != null && name.equals(this.activeStageName)) {
			return;
		}
		this.activeStageName = name;
		try {
			if (this.activeStageName != null) {
				this.activeStage = IStage.getIStage((Stage) CompileManager
						.getCompileManager().getClassLoader().loadClass(
								this.activeStageName).newInstance());
			}
		} catch (Throwable th) {
			this.activeStage = null;
		}

		TheaterObservable.getObservable().importantStateChange();
	}

	/**
	 * Ordnet dem Stck eine neue Bhne zu; es wird gleichzeitig auch der
	 * ActiveStageName gendert
	 * 
	 * @param stage
	 */
	public void setActiveStage(IStage stage) {
		if (stage == this.activeStage) {
			return;
		}
		this.activeStage = stage;
		if (stage != null) {
			Stage st = IStage.getStage(stage);
			this.activeStageName = st.getClass().getSimpleName();
		} else {
			this.activeStageName = null;
		}
		TheaterObservable.getObservable().importantStateChange();
	}

	/**
	 * Liefert den Namen der aktiven Bhne
	 * 
	 * @return
	 */
	public String getActiveStageName() {
		return this.activeStageName;
	}

	/**
	 * Liefert die aktuelle Bhne (kann auch null sein)
	 * 
	 * @return
	 */
	public IStage getActiveStage() {
		return this.activeStage;
	}

	/**
	 * Setzt den Namen einer neuen aktiven Performance und ldt gleichzeitig
	 * auch das entsprechende Objekt
	 * 
	 * @param name
	 */
	public void setActivePerformanceName(String name) {
		this.activePerformanceName = name;
		this.loadActivePerformance();
	}

	/**
	 * ldt das aktive Performance Objekt
	 */
	public void loadActivePerformance() {
		try {
			if (this.activePerformanceName == null) {
				this.activePerformance = IPerformance.getDefaultPerformance();
			} else {
				IPerformance newPerf = IPerformance
						.getIPerformance((Performance) CompileManager
								.getCompileManager().getClassLoader()
								.loadClass(this.activePerformanceName)
								.newInstance());
				newPerf.setSimulationSpeed(this.activePerformance
						.getSimulationSpeed());
				this.activePerformance = newPerf;
			}
		} catch (Throwable exc) {
			exc.printStackTrace();
		}
	}

	/**
	 * Liefert die aktive Performance (kann nicht null sein)
	 * 
	 * @return
	 */
	public IPerformance getActivePerformance() {
		return this.activePerformance;
	}

	/**
	 * Liefert den Namen der aktiven Performance (kann auch null sein, wenn es
	 * sich um die DefaultPerformance handelt)
	 * 
	 * @return
	 */
	public String getActivePerformanceName() {
		return this.activePerformanceName;
	}

	/**
	 * Liefert das Verzeichnis des Stckes
	 * 
	 * @return
	 */
	public String getDirectory() {
		return this.directory;
	}

	/**
	 * Ldt und setzt die neue aktive Bhne
	 * 
	 * @return
	 */
	public IStage reset() {
		if (!Application.isStandAlone()) {
			try {
				// damit auch static-Initializer erneut ausgefhrt werden
				CompileManager.getCompileManager().compile(Play.getPlay());

				ClassLoader cl = CompileManager.getCompileManager()
						.getClassLoader();
				String sName = this.getActiveStageName();
				if (sName != null) {
					TheaterObservable.getObservable().setEnabled(false);
					IStage s = IStage.getIStage((Stage) cl.loadClass(sName)
							.newInstance());
					TheaterObservable.getObservable().setEnabled(true);
					this.setActiveStage(s);
				}
			} catch (Throwable exc) {
				// exc.printStackTrace();
				this.setActiveStage(null);
			}
			loadActivePerformance();
		} else {
			try {
				this.activePerformance = IPerformance.getDefaultPerformance();
				if (this.activePerformanceName != null) {
					this.activePerformance = IPerformance
							.getIPerformance((Performance) Class.forName(
									"data." + this.activePerformanceName)
									.newInstance());

				}
				this.activePerformance
						.setSimulationSpeed(this.activePerformance
								.getSimulationSpeed());
				TheaterObservable.getObservable().setEnabled(false);
				this.activeStage = IStage.getIStage((Stage) Class.forName(
						"data." + this.activeStageName).newInstance());
				TheaterObservable.getObservable().setEnabled(true);
			} catch (Throwable exc) {
				// exc.printStackTrace();
				this.setActiveStage(null);
			}
		}

		TheaterObservable.getObservable().importantStateChange();
		return this.getActiveStage();
	}

	public boolean isEventQueueActive() {
		synchronized (syncObject) {
			return isEventQueueActive;
		}
	}

	public void setEventQueueActive(boolean active) {
		synchronized (syncObject) {
			this.isEventQueueActive = active;
		}
	}

	/**
	 * Liefert alle Performance-Klassen des Stckes
	 * 
	 * @return
	 */
	public ArrayList<String> getPerformanceClasses() {
		return this.performances;
	}

	/**
	 * Fgt eine neue Performance-Klasse hinzu
	 * 
	 * @param name
	 */
	public void insertPerformanceClass(String name) {
		this.performances.add(name);
		ClassManager.getClassManager().addClass(
				new TheaterClass(name, Performance.class, this));
	}

	/**
	 * Entfernt eine Performance-Klasse
	 * 
	 * @param name
	 */
	public void removePerformanceClass(String name) {
		this.performances.remove(name);
		ClassManager.getClassManager().removeClass(name);
	}

	/**
	 * Liefert alle Bhnen-Klassen
	 * 
	 * @return
	 */
	public ArrayList<String> getStageClasses() {
		return this.stages;
	}

	/**
	 * Fgt eine neue Bhnen-Klasse hinzu
	 * 
	 * @param name
	 */
	public void insertStageClass(String name) {
		this.stages.add(name);
		ClassManager.getClassManager().addClass(
				new TheaterClass(name, Stage.class, this));
	}

	/**
	 * Entfernt eine Bhnen-Klasse
	 * 
	 * @param name
	 */
	public void removeStageClass(String name) {
		this.stages.remove(name);
		ClassManager.getClassManager().removeClass(name);
	}

	/**
	 * Liefert alle Actor-Klassen
	 * 
	 * @return
	 */
	public ArrayList<String> getActorClasses() {
		return this.actors;
	}

	/**
	 * Fgt eine neue Actor-Klasse hinzu
	 * 
	 * @param name
	 */
	public void insertActorClass(String name) {
		this.actors.add(name);
		ClassManager.getClassManager().addClass(
				new TheaterClass(name, Actor.class, this));
	}

	/**
	 * Entfernt eine Actor-Klasse
	 * 
	 * @param name
	 */
	public void removeActorClass(String name) {
		this.actors.remove(name);
		ClassManager.getClassManager().removeClass(name);
	}

	/**
	 * Liefert alle Prop-Klassen
	 * 
	 * @return
	 */
	public ArrayList<String> getPropClasses() {
		return this.props;
	}

	/**
	 * Fgt eine neue Prop-Klasse hinzu
	 * 
	 * @param name
	 */
	public void insertPropClass(String name) {
		this.props.add(name);
		ClassManager.getClassManager().addClass(
				new TheaterClass(name, Prop.class, this));
	}

	/**
	 * Entfernt eine Prop-Klasse
	 * 
	 * @param name
	 */
	public void removePropClass(String name) {
		this.props.remove(name);
		ClassManager.getClassManager().removeClass(name);
	}

	/**
	 * Liefert alle Other-Klassen
	 * 
	 * @return
	 */
	public ArrayList<String> getOtherClasses() {
		return this.others;
	}

	/**
	 * Fgt eine neue Other-Klasse hinzu
	 * 
	 * @param name
	 */
	public void insertOtherClass(String name) {
		this.others.add(name);
		ClassManager.getClassManager().addClass(
				new TheaterClass(name, null, this));
	}

	/**
	 * Entfernt eine Other-Klasse
	 * 
	 * @param name
	 */
	public void removeOtherClass(String name) {
		this.others.remove(name);
		ClassManager.getClassManager().removeClass(name);
	}

	/**
	 * Liefert den Editor des Info-Textes
	 * 
	 * @return
	 */
	public Editor getInfoEditor() {
		return this.infoEditor;
	}

	/**
	 * Speichert ein Theaterstck in einer XML-Datei ab
	 */
	public void save() {
		try {
			File dir = new File(this.directory);
			File xmlFile = new File(dir.getAbsolutePath() + File.separatorChar
					+ "theater.xml");

			XMLOutputFactory factory = XMLOutputFactory.newInstance();
			XMLStreamWriter writer = factory.createXMLStreamWriter(
					new FileOutputStream(xmlFile.getAbsolutePath()), "utf-8");
			// Der XML-Header wird erzeugt
			writer.writeStartDocument("utf-8", "1.0");
			writer.writeCharacters("\n");

			// gibt Probleme mit Applets
			// writer.writeDTD("<!DOCTYPE theater SYSTEM \"theater.dtd\">");
			// writer.writeCharacters("\n");

			writer.writeStartElement("theater");
			writer.writeCharacters("\n");
			writer.writeStartElement("classes");
			writer.writeCharacters("\n");

			for (int i = 0; i < this.performances.size(); i++) {
				writer.writeStartElement("performance");
				writer.writeAttribute("name", this.performances.get(i));
				writer.writeEndElement(); // performance
				writer.writeCharacters("\n");
			}

			for (int i = 0; i < this.stages.size(); i++) {
				writer.writeStartElement("stage");
				writer.writeAttribute("name", this.stages.get(i));
				writer.writeEndElement(); // stage
				writer.writeCharacters("\n");
			}

			for (int i = 0; i < this.actors.size(); i++) {
				writer.writeStartElement("actor");
				writer.writeAttribute("name", this.actors.get(i));
				writer.writeEndElement(); // actor
				writer.writeCharacters("\n");
			}

			for (int i = 0; i < this.props.size(); i++) {
				writer.writeStartElement("prop");
				writer.writeAttribute("name", this.props.get(i));
				writer.writeEndElement(); // prop
				writer.writeCharacters("\n");
			}

			for (int i = 0; i < this.others.size(); i++) {
				writer.writeStartElement("other");
				writer.writeAttribute("name", this.others.get(i));
				writer.writeEndElement(); // other
				writer.writeCharacters("\n");
			}

			writer.writeEndElement(); // classes
			writer.writeCharacters("\n");

			if (this.activePerformanceName != null) {
				writer.writeStartElement("activeperformance");
				writer.writeAttribute("name", this.activePerformanceName);
				writer.writeEndElement(); // activeperformance
				writer.writeCharacters("\n");
			}

			if (this.activeStageName != null) {
				writer.writeStartElement("activestage");
				writer.writeAttribute("name", this.activeStageName);
				writer.writeEndElement(); // activestage
				writer.writeCharacters("\n");
			}

			writer.writeEndElement(); // theater
			writer.writeCharacters("\n");
			writer.writeEndDocument();
			writer.writeCharacters("\n");
			writer.close();

		} catch (Throwable exc) {
			exc.printStackTrace();
		}

	}

	/**
	 * Eruegt ein neues leeres Theaterstck
	 * 
	 * @param dirName
	 */
	protected void create(String dirName) {
		try {
			File dir = new File(dirName);

			// create dir
			dir.mkdir();

			// create dtd-file
			File dtdFile = new File(dir.getAbsolutePath() + File.separatorChar
					+ "theater.dtd");
			dtdFile.createNewFile();
			IO.copyResTextFile("theater.dtd", dtdFile.getAbsolutePath());

			// create play-file
			File xmlFile = new File(dir.getAbsolutePath() + File.separatorChar
					+ "theater.xml");
			xmlFile.createNewFile();

			// create info-file
			File infoFile = new File(dir.getAbsolutePath() + File.separatorChar
					+ "Info.txt");
			infoFile.createNewFile();
			IO.copyResTextFile("Info.txt", infoFile.getAbsolutePath(),
					"Info.txt");

			// create images dir
			File imagesDir = new File(dir.getAbsolutePath()
					+ File.separatorChar + "images");
			imagesDir.mkdir();

			// create sounds dir
			File soundsDir = new File(dir.getAbsolutePath()
					+ File.separatorChar + "sounds");
			soundsDir.mkdir();

		} catch (Throwable exc) {
			exc.printStackTrace();
		}
	}

	/**
	 * Ldt ein Theaterstck aus einer XML-Datei
	 * 
	 * @param xmlFile
	 */
	protected void load(File xmlFile) {
		try {
			InputStream in = new FileInputStream(xmlFile);
			XMLInputFactory factory = XMLInputFactory.newInstance();
			XMLEventReader parser = factory.createXMLEventReader(in);
			while (parser.hasNext()) {
				XMLEvent event = parser.nextEvent();
				switch (event.getEventType()) {
				case XMLStreamConstants.START_DOCUMENT:
					break;
				case XMLStreamConstants.END_DOCUMENT:
					parser.close();
					break;
				case XMLStreamConstants.START_ELEMENT:
					StartElement element = event.asStartElement();
					if ("performance".equals(element.getName().getLocalPart())) {
						this.insertPerformanceClass(element.getAttributeByName(
								new QName("name")).getValue());
					} else if ("stage".equals(element.getName().getLocalPart())) {
						this.insertStageClass(element.getAttributeByName(
								new QName("name")).getValue());
					} else if ("actor".equals(element.getName().getLocalPart())) {
						this.insertActorClass(element.getAttributeByName(
								new QName("name")).getValue());
					} else if ("prop".equals(element.getName().getLocalPart())) {
						this.insertPropClass(element.getAttributeByName(
								new QName("name")).getValue());
					} else if ("other".equals(element.getName().getLocalPart())) {
						this.insertOtherClass(element.getAttributeByName(
								new QName("name")).getValue());
					} else if ("activestage".equals(element.getName()
							.getLocalPart())) {
						this.activeStageName = element.getAttributeByName(
								new QName("name")).getValue();
					} else if ("activeperformance".equals(element.getName()
							.getLocalPart())) {
						this.activePerformanceName = element
								.getAttributeByName(new QName("name"))
								.getValue();
					}
					break;
				default:
					break;
				}
			}

		} catch (Throwable exc) {
			exc.printStackTrace();
		}
	}

}

class SAXHandler extends DefaultHandler {

	Play play;

	SAXHandler(Play play) {
		this.play = play;
	}

	void load(String xmlFile) {  // fr Standalone

		SAXParserFactory factory = SAXParserFactory.newInstance();
		try {

			SAXParser saxParser = factory.newSAXParser();
			saxParser.parse(SAXHandler.class.getClassLoader()
					.getResourceAsStream(xmlFile), this);
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}
	
	void load(File xmlFile) {

		SAXParserFactory factory = SAXParserFactory.newInstance();
		try {

			SAXParser saxParser = factory.newSAXParser();
			saxParser.parse(xmlFile, this);
		} catch (Throwable t) {
			t.printStackTrace();
		}
	}

	public void startElement(String namespaceURI, String simpleName,
			String qName, Attributes attrs) throws SAXException {
		String nameValue = null;
		if (attrs != null) {
			for (int i = 0; i < attrs.getLength(); i++) {
				String aName = attrs.getLocalName(i); 

				if ("name".equals(aName)) {
					nameValue = attrs.getValue(i);
					break;
				}
			}
		}
		if ("".equals(simpleName)) {
			simpleName = qName;
		}

		if ("performance".equals(simpleName)) {
			play.insertPerformanceClass(nameValue);
		} else if ("stage".equals(simpleName)) {
			play.insertStageClass(nameValue);
		} else if ("actor".equals(simpleName)) {
			play.insertActorClass(nameValue);
		} else if ("prop".equals(simpleName)) {
			play.insertPropClass(nameValue);
		} else if ("other".equals(simpleName)) {
			play.insertOtherClass(nameValue);
		} else if ("activestage".equals(simpleName)) {
			play.activeStageName = nameValue;
		} else if ("activeperformance".equals(simpleName)) {
			play.activePerformanceName = nameValue;
		}
	}

}
