package util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.Enumeration;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

public class ZipArchives {

	/**
	 * entpackt (rekursiv) eine zip-Datei in ein angegebenes Verzeichnis
	 * 
	 * @param archive
	 *            die existierende zip-Datei
	 * @param destDir
	 *            das Verzeichnis, in das entpackt werden soll (wird erzeugt,
	 *            falls es nicht existiert)
	 * @throws IOException
	 */
	public static void unzip(File archive, File destDir) throws IOException {
		if (!destDir.exists()) {
			destDir.mkdirs();
		}

		ZipFile zipFile = new ZipFile(archive);
		Enumeration<? extends ZipEntry> entries = zipFile.entries();

		byte[] buffer = new byte[4096];
		int len;
		while (entries.hasMoreElements()) {
			ZipEntry entry = entries.nextElement();

			String entryFileName = entry.getName();

			File dir = ZipArchives.buildDirectoryHierarchyFor(entryFileName,
					destDir);
			if (!dir.exists()) {
				dir.mkdirs();
			}

			if (!entry.isDirectory()) {
				try {
					BufferedOutputStream bos = new BufferedOutputStream(
							new FileOutputStream(new File(destDir,
									entryFileName)));

					BufferedInputStream bis = new BufferedInputStream(zipFile
							.getInputStream(entry));

					while ((len = bis.read(buffer)) > 0) {
						bos.write(buffer, 0, len);
					}

					bos.flush();
					bos.close();
					bis.close();
				} catch (Throwable th) {
					th.printStackTrace();
				}
			}
		}
	}

	/**
	 * Packt mehrere Verzeichnisse in eine zip-Datei
	 * 
	 * @param zipFile
	 * @param dirs2zip
	 * @throws IOException
	 */
	public static void zipDirs(String zipFile, String... dirs2zip)
			throws IOException {

		ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(zipFile));
		for (String dir : dirs2zip) {
			ZipArchives.zipDir(dir, zos);
		}
		zos.close();
	}

	/**
	 * Erzeugt eine zip-Datei und packt ein Verzeichnis (rekursiv) in die
	 * zip-Datei
	 * 
	 * @param dir2zip
	 *            das zu verpackende Verzeichnis
	 * @param zipFile
	 *            Name der neuen zip-Datei
	 * @throws IOException
	 */
	public static void zipDir(String dir2zip, String zipFile)
			throws IOException {

		ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(zipFile));
		ZipArchives.zipDir(dir2zip, zos);
		zos.close();
	}

	private static void zipDir(String dir2zip, ZipOutputStream zos)
			throws IOException {

		File zipDir = new File(dir2zip);
		String[] dirList = zipDir.list();
		byte[] readBuffer = new byte[2156];
		int bytesIn = 0;

		System.out.println("dir = " + dir2zip);
		for (int i = 0; i < dirList.length; i++) {
			File f = new File(zipDir, dirList[i]);
			if (f.isDirectory()) {

				String filePath = f.getPath();
				ZipArchives.zipDir(filePath, zos);

				continue;
			}

			FileInputStream fis = new FileInputStream(f);

			ZipEntry anEntry = new ZipEntry(entryName(f.getPath()));
			zos.putNextEntry(anEntry);

			while ((bytesIn = fis.read(readBuffer)) != -1) {
				zos.write(readBuffer, 0, bytesIn);
			}

			fis.close();
		}

	}


	/**
	 * Packt mehrere Verzeichnisse in eine zip-Datei
	 * 
	 * @param zipFile
	 * @param dirs2zip
	 * @throws IOException
	 */
	public static void jarDirs(String jarFile, String manifest_version,
			String main_class, String... dirs2zip) throws IOException {
		Manifest man = new Manifest();
		Attributes global = man.getMainAttributes();

		// Create the manifest.
		global.put(Attributes.Name.MANIFEST_VERSION, manifest_version);
		global.put(Attributes.Name.MAIN_CLASS, main_class);

		final JarOutputStream zos = new JarOutputStream(new FileOutputStream(
				jarFile), man);

		for (String dir : dirs2zip) {
			ZipArchives.jarDir(dir, zos);
		}
		zos.close();
	}

	/**
	 * Erzeugt eine zip-Datei und packt ein Verzeichnis (rekursiv) in die
	 * zip-Datei
	 * 
	 * @param dir2zip
	 *            das zu verpackende Verzeichnis
	 * @param zipFile
	 *            Name der neuen zip-Datei
	 * @throws IOException
	 */
	public static void jarDir(String dir2zip, String zipFile)
			throws IOException {

		JarOutputStream zos = new JarOutputStream(new FileOutputStream(zipFile));
		ZipArchives.jarDir(dir2zip, zos);
		zos.close();
	}

	private static void jarDir(String dir2zip, JarOutputStream zos)
			throws IOException {

		File zipDir = new File(dir2zip);
		String[] dirList = zipDir.list();
		byte[] readBuffer = new byte[2156];
		int bytesIn = 0;

		for (int i = 0; i < dirList.length; i++) {
			File f = new File(zipDir, dirList[i]);
			if (f.isDirectory()) {

				String filePath = f.getPath();
				ZipArchives.jarDir(filePath, zos);

				continue;
			}

			FileInputStream fis = new FileInputStream(f);

			JarEntry anEntry = new JarEntry(entryName(f.getPath()));
			zos.putNextEntry(anEntry);

			while ((bytesIn = fis.read(readBuffer)) != -1) {
				zos.write(readBuffer, 0, bytesIn);
			}

			fis.close();
		}

	}

	/**
	 * Fgt Dateien oder (rekursiv) Verzeichnisse in eine zip-Datei hinzu
	 * 
	 * @param dir
	 *            Verzeichnis oder Datei, das hinzugefgt werden soll
	 * @param archiveFile
	 *            existierende zip-Datei
	 * @throws IOException
	 */
	public static void addFilesToZip(String dir, File archiveFile)
			throws IOException {

		File tmpFile = File.createTempFile(String.valueOf(System
				.currentTimeMillis()), "zip");

		ZipFile zipSrc = new ZipFile(archiveFile);

		ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(tmpFile));

		byte[] buffer = new byte[32756];
		int len = 0;

		Enumeration<? extends ZipEntry> srcEntries = zipSrc.entries();
		while (srcEntries.hasMoreElements()) {
			ZipEntry entry = srcEntries.nextElement();

			ZipEntry newEntry = new ZipEntry(entry.getName());

			zos.putNextEntry(newEntry);

			InputStream is = zipSrc.getInputStream(entry);

			while ((len = is.read(buffer)) > 0) {
				zos.write(buffer, 0, len);
			}

			zos.flush();
			is.close();
		}

		ZipArchives.addFiles(new File(dir), zos);

		zos.finish();
		zos.close();
		zipSrc.close();
		archiveFile.delete();
		tmpFile.renameTo(archiveFile);
	}

	private static void addFiles(File file, ZipOutputStream zos) {
		if (file.isDirectory()) {
			String[] files = file.list();
			for (String f : files) {
				ZipArchives.addFiles(new File(file, f), zos);
			}
		} else {
			ZipArchives.addFile(file, zos);
		}
	}

	private static void addFile(File file, ZipOutputStream zos) {
		try {
			ZipEntry newEntry = new ZipEntry(entryName(file.getPath()));
			zos.putNextEntry(newEntry);

			InputStream is = new FileInputStream(file);

			byte[] buffer = new byte[32756];
			int len = 0;
			while ((len = is.read(buffer)) > 0) {
				zos.write(buffer, 0, len);
			}

			is.close();
			zos.closeEntry();
		} catch (Exception exc) {
			exc.printStackTrace();
		}
	}

	private static File buildDirectoryHierarchyFor(String entryName,
			File destDir) {
		int lastIndex = entryName.lastIndexOf('/');
		String internalPathToEntry = entryName.substring(0, lastIndex + 1);
		return new File(destDir, internalPathToEntry);
	}

	static String entryName(String name) {
		return name.replace(File.separatorChar, '/');
	}

}
