package editor;

import java.util.HashMap;

/**
 * Lexikalische Analyse eines Java-Dokumentes.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class JavaLexer {
	/**
	 * Diese Schluesselwoerter kann der Lexer erkennen.
	 */
	public static final String[] KEYWORDS = { "abstract", "double", "int",
			"strictfp", "boolean", "else", "interface", "super", "break",
			"extends", "long", "switch", "byte", "final", "native",
			"synchronized", "case", "finally", "new", "this", "catch", "float",
			"package", "throw", "char", "for", "private", "throws", "class",
			"goto", "protected", "transient", "const", "if", "public", "try",
			"continue", "implements", "return", "void", "default", "import",
			"short", "volatile", "do", "instanceof", "static", "while" };

	/**
	 * Die Schluesselwoerter werden aus Performanzgruenden in einen HashMap
	 * gepackt.
	 */
	static HashMap<String, String> keywords;

	public static final int PLAIN = 0;
	public static final int COMMENT = 1;
	public static final int LITERAL = 2;
	public static final int KEYWORD = 3;
	public static final int WHITESPACE = 4;

	/**
	 * Der zu erkennende Text.
	 */
	protected String text;

	/**
	 * Die aktuelle Position im Text.
	 */
	protected int pos, off;

	/**
	 * Diese Methode packt die Schluesselwoerter in die HashMap.
	 */
	public static void init() {
		if (JavaLexer.keywords != null) {
			return;
		}
		JavaLexer.keywords = new HashMap<String, String>();
		for (int i = 0; i < JavaLexer.KEYWORDS.length; i++) {
			JavaLexer.keywords
					.put(JavaLexer.KEYWORDS[i], JavaLexer.KEYWORDS[i]);
		}
	}

	/**
	 * Der Konstruktor des HamsterLexers. Sorgt dafuer, dass init() aufgerufen
	 * wird.
	 */
	public JavaLexer() {
		JavaLexer.init();
	}

	/**
	 * Initialisiert den Lexer mit einem neuen Text und setzt gleichzeitig
	 * Position und Offset.
	 * 
	 * @param pos
	 *            Die neue Position
	 * @param offset
	 *            Der neue Offset
	 * @param text
	 *            Der neue Text
	 */
	public void init(int pos, int offset, String text) {
		this.pos = pos;
		this.off = offset;
		this.text = text;
	}

	/**
	 * Die LL(k) Methode eines Lexers.
	 * 
	 * @param k
	 *            lookahead
	 * @return Das Symbol im Abstand k
	 */
	private char LL(int k) {
		if (this.pos + k < this.text.length()) {
			return this.text.charAt(this.pos + k);
		} else {
			return (char) -1;
		}
	}

	/**
	 * Erkennt ein Leerzeichen
	 */
	private void consumeWhiteSpace() {
		this.pos++;
	}

	/**
	 * Erkennt einen Bezeichner
	 */
	private void consumeIdentifier() {
		this.pos++;
		while (Character.isJavaIdentifierPart(this.LL(0))) {
			this.pos++;
		}
	}

	/**
	 * Erkennt einen einzeiligen Kommentar
	 */
	private void consumeSingleLineComment() {
		this.pos += 2;
		while (this.LL(0) != '\n' && this.LL(0) != '\r'
				&& this.LL(0) != (char) -1) {
			this.pos++;
		}
		if (this.LL(0) == '\r' && this.LL(1) == '\n') {
			this.pos += 2;
		} else if (this.LL(0) == '\n') {
			this.pos++;
		} else if (this.LL(0) == '\r') {
			this.pos++;
		}
	}

	private void consumeMultiLineComment() {
		this.pos += 2;
		while ((this.LL(0) != '*' || this.LL(1) != '/')
				&& this.LL(0) != (char) -1) {
			this.pos++;
		}
		if (this.LL(0) == '*' && this.LL(1) == '/') {
			this.pos += 2;
		}
	}

	/**
	 * Erkennt ein einzelnes Zeichen
	 */
	private void consumeCharacter() {
		this.pos++;
	}

	/**
	 * Erkennt einen String-Literal
	 */
	private void consumeStringLiteral() {
		this.pos++;
		while (this.LL(0) != (char) -1 && this.LL(0) != '\"'
				&& this.LL(0) != '\r' && this.LL(0) != '\n') {
			if (this.LL(0) == '\\' && this.LL(1) == '\"') {
				this.pos++;
			}
			this.pos++;
		}
		if (this.LL(0) == '\r' && this.LL(1) == '\n') {
			this.pos += 2;
		} else if (this.LL(0) == '\n' || this.LL(0) == '\r') {
			this.pos++;
		} else if (this.LL(0) == '\"') {
			this.pos++;
		}
	}

	/**
	 * Erkennt einen Character-Literal
	 */
	private void consumeCharacterLiteral() {
		this.pos++;
		while (this.LL(0) != (char) -1 && this.LL(0) != '\''
				&& this.LL(0) != '\r' && this.LL(0) != '\n') {
			if (this.LL(0) == '\\' && this.LL(1) == '\'') {
				this.pos++;
			}
			this.pos++;
		}
		if (this.LL(0) == '\r' && this.LL(1) == '\n') {
			this.pos += 2;
		} else if (this.LL(0) == '\n' || this.LL(0) == '\r') {
			this.pos++;
		} else if (this.LL(0) == '\'') {
			this.pos++;
		}
	}

	/**
	 * Erkennt das naechste Token im Text
	 * 
	 * @return Das naechste Token
	 */
	public JavaToken nextToken() {
		if (!this.ready()) {
			return null;
		}
		int oldPos = this.pos;
		int type = JavaLexer.PLAIN;
		if (this.LL(0) == ' ' || this.LL(0) == '\t' || this.LL(0) == '\n'
				|| this.LL(0) == '\r') {
			this.consumeWhiteSpace();
			type = JavaLexer.WHITESPACE;
		} else if (Character.isJavaIdentifierStart(this.LL(0))) {
			this.consumeIdentifier();
		} else if (this.LL(0) == '/') {
			if (this.LL(1) == '/') {
				this.consumeSingleLineComment();
				type = JavaLexer.COMMENT;
			} else if (this.LL(1) == '*') {
				this.consumeMultiLineComment();
				type = JavaLexer.COMMENT;
			} else {
				this.consumeCharacter();
			}
		} else if (this.LL(0) == '\"') {
			this.consumeStringLiteral();
			type = JavaLexer.LITERAL;
		} else if (this.LL(0) == '\'') {
			this.consumeCharacterLiteral();
			type = JavaLexer.LITERAL;
		} else {
			this.consumeCharacter();
		}
		String t = this.text.substring(oldPos, this.pos);
		if (type == JavaLexer.PLAIN) {
			if (JavaLexer.keywords.get(t) != null) {
				type = JavaLexer.KEYWORD;
			}
		}
		return new JavaToken(t, this.off + oldPos, type);
	}

	/**
	 * Gibt an, ob noch weitere Token existieren.
	 * 
	 * @return true, wenn der Lexer noch nicht am Ende des Textes ist.
	 */
	public boolean ready() {
		return this.pos < this.text.length();
	}
}