package theater;

import model.ClassManager;
import theater_intern.IComponent;
import theater_intern.IStage;

/**
 * Die Klasse Component definiert Methoden zur Verwaltung von Akteuren und
 * Requisiten, die sie an die von ihr abgeleiteten Klassen Actor und Prop
 * vererbt. Die Klassen Actor und Prop unterscheiden sich nur dadurch, dass von
 * Actor abgeleitete Klassen die Methode run berschreiben knnen. Bei
 * Unterklassen von Prop wird dies dadurch verhindert, dass die run-Methode in
 * der Klasse Prop als leere final-Methode definiert wird.
 * <p>
 * </p>
 * Die wichtigsten Methoden der Klasse Component sind Methoden, um Akteuren und
 * Requisiten ein Icon zuzuordnen und sie auf der Bhne bewegen, also
 * umplatzieren oder bspw. rotieren zu knnen. Weiterhin ist eine Menge von
 * Getter-Methoden definiert, um zum einen ihren Zustand abfragen und zum
 * anderen das Bhnen-Objekt ermitteln zu knnen, dem sie u.U. zugeordnet sind.
 * <p>
 * </p>
 * Wie die Klasse Stage enthlt die Klasse Component darber hinaus
 * Kollisionserkennungsmethoden zum Entdecken von Kollisionen der entsprechenden
 * Komponente mit anderen Komponenten sowie Methoden zur Verwaltung von Maus-
 * und Tastatur-Events. Die Klasse Component implementiert das Interface
 * PixelArea, so dass Akteure und Requisiten unmittelbar selbst in die
 * Kollisionserkennung mit einbezogen werden knnen.
 * 
 * @author Dietrich Boles, Universitt Oldenburg, Germany
 * @version 1.0 (08.09.2009)
 * 
 */
public class Component extends Object implements PixelArea {

	/**
	 * Standardmiger Wert der Z-Koordinate
	 */
	public final static int DEF_Z_COORDINATE = 0;

	/**
	 * intern genutztes Objekt
	 */
	IComponent comp;

	/**
	 * Konstruktor, der eine neue Komponente als Akteur oder Requisite
	 * initialisiert.
	 * 
	 * @param isActor
	 *            true, falss es sich um einen Akteur handelt; false, falls es
	 *            sich um eine Requisite handelt
	 */
	protected Component(boolean isActor) {
		this.comp = new IComponent(isActor);
		IComponent.newCompIComp(this, this.comp);
		this.comp.setImage(ClassManager.getClassManager().getClass(
				this.getClass().getSimpleName()).getImage());
	}

	/**
	 * Muss berschrieben werden und die Actor-spezifischen Aktionen enthalten
	 */
	public void run() {
	}

	/**
	 * Liefert das Objekt der Bhne, auf dem sich die Komponente gerade befindet
	 * oder null, falls sich die Komponente aktuell auf keiner Bhne befindet
	 * 
	 * @return die Bhne, auf der sich die Komponente gerade befindet
	 */
	public Stage getStage() {
		return IStage.getStage(this.comp.getStage());
	}

	/**
	 * Ordnet der Komponente ein Icon zu, durch das sie auf der Bhne
	 * reprsentiert wird. Erlaubt sind Bilder der Formate gif, jpg und png.
	 * 
	 * @param filename
	 *            Name der Bilddatei; die Datei muss sich im Unterverzeichnis
	 *            images des Theaterstck-Verzeichnisses befinden
	 * @throws IllegalArgumentException
	 *             wird geworfen, wenn die angegebene Datei keine gltige
	 *             lesbare Bilddatei ist
	 */
	public void setImage(String filename) throws IllegalArgumentException {
		this.comp.setImage(filename);
	}

	/**
	 * Ordnet der Komponente ein TheaterImage als Icon zu, durch das sie auf der
	 * Bhne reprsentiert wird.
	 * 
	 * @param image
	 *            das TheaterImage, das als Icon verwendet werden soll
	 */
	public void setImage(TheaterImage image) {
		this.comp.setImage(image);
	}

	/**
	 * Liefert das Icon der Komponente als TheaterImage-Objekt. Wurde kein Icon
	 * zugeordnet, wird null geliefert.
	 * 
	 * @return das Icon der Komponente als TheaterImage-Objekt
	 */
	public TheaterImage getImage() {
		return this.comp.getImage();
	}

	/**
	 * Mit Hilfe der add-Methode der Klasse Stage kann eine Komponente auf einer
	 * Bhne platziert werden. Nach der erfolgreichen Platzierung wird von der
	 * Bhne diese Methode addedToStage fr die Komponente aufgerufen. Als
	 * Parameter wird dabei das Bhnenobjekt bergeben. Sollen fr eine
	 * Komponente bestimmte Aktionen ausgefhrt werden, sobald sie einer Bhne
	 * zugeordnet wird, muss die Methode entsprechend berschrieben werden.
	 * 
	 * @param stage
	 *            das Objekt, das die Bhne reprsentiert, auf die die
	 *            Komponente platziert wurde
	 */
	public void addedToStage(Stage stage) {
	}

	/**
	 * Mit Hilfe der add-Methode der Klasse Stage knnen Komponente in einer
	 * bestimmten Spalte und Reihe auf der Bhne platziert werden. Die Methode
	 * setLocation ermglicht die Umplatzierung der Komponente auf der Bhne.
	 * 
	 * @param newCol
	 *            die Spalte, in die die Komponente umplatziert werden soll
	 * @param newRow
	 *            die Reihe, in die die Komponente umplatziert werden soll *
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf einer
	 *             Bhne platziert ist
	 */
	public void setLocation(int newCol, int newRow)
			throws IllegalStateException {
		this.comp.setLocation(newCol, newRow);
	}

	/**
	 * Liefert die Spalte, in der sich die Komponente aktuell auf einer Bhne
	 * befindet.
	 * 
	 * @return die Spalte, in der sich die Komponente aktuell auf einer Bhne
	 *         befindet
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf einer
	 *             Bhne platziert ist
	 */
	public int getColumn() throws IllegalStateException {
		return this.comp.getColumn();
	}

	/**
	 * Liefert die Reihe, in der sich die Komponente aktuell auf einer Bhne
	 * befindet.
	 * 
	 * @return die Reihe, in der sich die Komponente aktuell auf einer Bhne
	 *         befindet
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf einer
	 *             Bhne platziert ist
	 */
	public int getRow() throws IllegalStateException {
		return this.comp.getRow();
	}

	/**
	 * Liefert die Zelle, in der sich die Komponente aktuell auf einer Bhne
	 * befindet.
	 * 
	 * @return die Zelle, in der sich die Komponente aktuell auf einer Bhne
	 *         befindet
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf einer
	 *             Bhne platziert ist
	 */
	public Cell getCell() throws IllegalStateException {
		return new Cell(this.comp.getColumn(), this.comp.getRow());
	}

	/**
	 * ndert die z-Koordinate der Komponente, mit der die Zeichenreihenfolge
	 * von Komponenten beeinflusst werden kann. Je hher der Wert der
	 * z-Koordinate einer Komponente ist, umso weiter gelangt das Icon der
	 * Komponente auf der Bhne in den Vordergrund. Die Zeichenreihenfolge von
	 * Komponenten mit gleicher z-Koordinate ist undefiniert. Standardmig hat
	 * die z-Koordinate einer Komponente den Wert DEF_Z_COORDINATE.
	 * 
	 * @param newZ
	 *            die neue z-Koordinate der Komponente
	 */
	public void setZCoordinate(int newZ) {
		this.comp.setZCoordinate(newZ);
	}

	/**
	 * Liefert die aktuelle z-Koordinate der Komponente.
	 * 
	 * @return die aktuelle z-Koordinate der Komponente
	 */
	public int getZCoordinate() {
		return this.comp.getZCoordinate();
	}

	/**
	 * Legt den Rotationswinkel fest, mit der das Icon der Komponente gezeichnet
	 * werden soll. Die Drehung erfolgt im Uhrzeigersinn. Standardmig betrgt
	 * der Rotationswinkel 0.
	 * <p>
	 * </p>
	 * Achtung: Der Rotationswinkel hat keinen Einfluss auf die Weite und Hhe
	 * eines einer Komponente zugeordneten Icons. Diese werden immer auf der
	 * Grundlage eines Rotationswinkels von 0 berechnet.
	 * 
	 * @param rotation
	 *            der neue Rotationswinkel der Komponente
	 */
	public void setRotation(int rotation) {
		this.comp.setRotation(rotation);
	}

	/**
	 * Liefert den aktuellen Rotationswinkel der Komponente.
	 * 
	 * @return der aktuelle Rotationswinkel der Komponente
	 */
	public int getRotation() {
		return this.comp.getRotation();
	}

	/**
	 * ber diese Methode knnen Komponenten sichtbar bzw. unsichtbar gemacht
	 * werden. In mglichen Kollisionsabfragen werden allerdings auch
	 * unsichtbare Komponenten mit einbezogen.
	 * 
	 * @param visible
	 *            falls true, wird die Komponente sichtbar; falls false, wird
	 *            sie unsichtbar
	 */
	public void setVisible(boolean visible) {
		this.comp.setVisible(visible);
	}

	/**
	 * Liefert die Sichtbarkeit der Komponente.
	 * 
	 * @return true, falls die Komponente sichtbar ist; ansonsten false
	 */
	public boolean isVisible() {
		return this.comp.isVisible();
	}

	/**
	 * berprft, ob der angegebene Punkt mit den Koordinaten x und y innerhalb
	 * des Icons der Komponente liegt.
	 * 
	 * @param x
	 *            x-Koordinate des Punktes
	 * @param y
	 *            y-Koordinate des Punktes
	 * @return true, falls der angegebene Punkt innerhalb des Icons der
	 *         Komponente liegt *
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf der
	 *             Bhne platziert ist
	 * 
	 * @see theater.PixelArea#contains(int, int)
	 */
	public boolean contains(int x, int y) throws IllegalStateException {
		return this.comp.contains(x, y);
	}

	/**
	 * berprft, ob das Icon der Komponente vollstndig innerhalb der
	 * angegebenen PixelArea liegt.
	 * 
	 * @param area
	 *            das Gebiet, das berprft werden soll (darf nicht null sein)
	 * @return true, falls das Icon der Komponente vollstndig innerhalb der
	 *         angegebenen PixelArea liegt *
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf der
	 *             Bhne platziert ist
	 * 
	 * @see theater.PixelArea#isInside(theater.PixelArea)
	 */
	public boolean isInside(PixelArea area) throws IllegalStateException {
		return this.comp.isInside(area);
	}

	/**
	 * berprft, ob das Icon der Komponente eine angegebene PixelArea
	 * schneidet.
	 * 
	 * @param area
	 *            das Gebiet, das berprft werden soll (darf nicht null sein)
	 * @return true, falls das Icon der Komponente die angegebenen PixelArea
	 *         schneidet
	 * @throws IllegalStateException
	 *             wird geworfen, wenn die Komponente aktuell nicht auf der
	 *             Bhne platziert ist
	 * 
	 * @see theater.PixelArea#intersects(theater.PixelArea)
	 */
	public boolean intersects(PixelArea area) throws IllegalStateException {
		return this.comp.intersects(area);
	}

	/**
	 * Legt fest, ob die Komponente Tastatur-Ereignisse behandeln soll.
	 * Standardmig ist dies der Fall.
	 * 
	 * @param handlingKeyEvents
	 *            true, falls die Komponente Tastatur-Ereignisse behandeln soll;
	 *            false andernfalls.
	 */
	public void setHandlingKeyEvents(boolean handlingKeyEvents) {
		this.comp.setHandlingKeyEvents(handlingKeyEvents);
	}

	/**
	 * berprft, ob die Komponente Tastatur-Ereignisse behandelt. Standardmig
	 * ist dies der Fall.
	 * 
	 * @return true, falls die Komponente Tastatur-Ereignisse behandelt
	 */
	public boolean isHandlingKeyEvents() {
		return this.comp.isHandlingKeyEvents();
	}

	/**
	 * Wird aufgerufen, wenn whrend die Bhne den Focus besitzt ein
	 * keyTyped-Event eingetreten ist. Soll eine Komponente auf keyTyped-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu keyTyped-Events finden sich in der Klasse
	 * java.awt.event.KeyListener. bergeben wird der Methode ein
	 * KeyInfo-Objekt, ber das Details zum eingetretenen Event abgefragt werden
	 * knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingKeyEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf keyTyped-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void keyTyped(KeyInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn whrend die Bhne den Focus besitzt ein
	 * keyPressed-Event eingetreten ist. Soll eine Komponente auf
	 * keyPressed-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu keyPressed-Events finden sich in
	 * der Klasse java.awt.event.KeyListener. bergeben wird der Methode ein
	 * KeyInfo-Objekt, ber das Details zum eingetretenen Event abgefragt werden
	 * knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingKeyEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf keyTyped-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void keyPressed(KeyInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn whrend die Bhne den Focus besitzt ein
	 * keyReleased-Event eingetreten ist. Soll eine Komponente auf
	 * keyReleased-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu keyReleased-Events finden sich
	 * in der Klasse java.awt.event.KeyListener. bergeben wird der Methode ein
	 * KeyInfo-Objekt, ber das Details zum eingetretenen Event abgefragt werden
	 * knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingKeyEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf keyTyped-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void keyReleased(KeyInfo e) {
	}

	/**
	 * Legt fest, ob die Komponente Maus-Ereignisse behandeln soll.
	 * Standardmig ist dies der Fall.
	 * 
	 * @param handlingMouseEvents
	 *            true, falls die Komponente Maus-Ereignisse behandeln soll;
	 *            false andernfalls.
	 */
	public void setHandlingMouseEvents(boolean handlingMouseEvents) {
		this.comp.setHandlingMouseEvents(handlingMouseEvents);
	}

	/**
	 * berprft, ob die Komponente Maus-Ereignisse behandelt. Standardmig ist
	 * dies der Fall.
	 * 
	 * @return true, falls die Komponente Maus-Ereignisse behandelt
	 */
	public boolean isHandlingMouseEvents() {
		return this.comp.isHandlingMouseEvents();
	}

	/**
	 * Wird aufgerufen, wenn ein mousePressed-Event auf der Komponente
	 * eingetreten ist, d.h. eine Maustaste gedrckt wird, whrend sich der
	 * Mauszeiger oberhalb des Icons der Komponente befindet. Soll eine
	 * Komponente auf mousePressed-Events reagieren, muss sie diese Methode
	 * entsprechend berschreiben. Genauere Informationen zu mousePressed-Events
	 * finden sich in der Klasse java.awt.event.MouseListener. bergeben wird
	 * der Methode ein MouseInfo-Objekt, ber das Details zum eingetretenen
	 * Event abgefragt werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf keyTyped-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mousePressed(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseReleased-Event auf der Bhne eingetreten
	 * ist, d.h. eine gedrckte Maustaste losgelassen wird, whrend sich der
	 * Mauszeiger ber dem Icon der Komponente befindet. Soll eine Komponente
	 * auf mouseReleased-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu mouseReleased-Events finden sich
	 * in der Klasse java.awt.event.MouseListener. bergeben wird der Methode
	 * ein MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf mouseReleased-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseReleased(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseClicked-Event auf der Komponente
	 * eingetreten ist, d.h. eine Maustaste geklickt wurde, whrend sich der
	 * Mauszeiger auf dem Icons der Komponente befindet. Soll eine Komponente
	 * auf mouseClicked-Events reagieren, muss sie diese Methode entsprechend
	 * berschreiben. Genauere Informationen zu mouseClicked-Events finden sich
	 * in der Klasse java.awt.event.MouseListener. bergeben wird der Methode
	 * ein MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf mouseClicked-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseClicked(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseDragged-Event auf der Komponente
	 * eingetreten ist, d.h. die Maus bei gedrckter Maustaste bewegt wurde,
	 * whrend sich der Mauszeiger auf dem Icon der Komponente befindet. Soll
	 * eine Komponente auf mouseDragged-Events reagieren, muss sie diese Methode
	 * entsprechend berschreiben. Genauere Informationen zu mouseDragged-Events
	 * finden sich in der Klasse java.awt.event.MouseMotionListener. bergeben
	 * wird der Methode ein MouseInfo-Objekt, ber das Details zum eingetretenen
	 * Event abgefragt werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf mouseDragged-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseDragged(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseMoved-Event auf der Komponente eingetreten
	 * ist, d.h. die Maus bewegt wurde, whrend sich der Mauszeiger auf dem Icon
	 * der Komponente befindet. Soll eine Komponente auf mouseMoved-Events
	 * reagieren, muss sie diese Methode entsprechend berschreiben. Genauere
	 * Informationen zu mouseMoved-Events finden sich in der Klasse
	 * java.awt.event.MouseMotionListener. bergeben wird der Methode ein
	 * MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf mouseMoved-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseMoved(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseEntered-Event auf der Komponente
	 * eingetreten ist, d.h. der Mauszeiger auf das Icon der Komponente gezogen
	 * wird. Soll eine Komponente auf mouseEntered-Events reagieren, muss sie
	 * diese Methode entsprechend berschreiben. Genauere Informationen zu
	 * mouseEntered-Events finden sich in der Klasse
	 * java.awt.event.MouseListener. bergeben wird der Methode ein
	 * MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf mouseEntered-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseEntered(MouseInfo e) {
	}

	/**
	 * Wird aufgerufen, wenn ein mouseExited-Event auf der Komponente
	 * eingetreten ist, d.h. der Mauszeiger das Icon der Komponente verlsst.
	 * Soll eine Komponente auf mouseExited-Events reagieren, muss sie diese
	 * Methode entsprechend berschreiben. Genauere Informationen zu
	 * mouseExited-Events finden sich in der Klasse
	 * java.awt.event.MouseListener. bergeben wird der Methode ein
	 * MouseInfo-Objekt, ber das Details zum eingetretenen Event abgefragt
	 * werden knnen.
	 * <p>
	 * </p>
	 * Die Methode wird nur aufgerufen, wenn der HandlingMouseEvents-Status
	 * gesetzt ist (was standardmig der Fall ist).
	 * <p>
	 * </p>
	 * Sowohl Komponenten als auch die Bhne knnen auf mouseExited-Events
	 * reagieren. Die Reihenfolge, in der eine Benachrichtigung erfolgt, richtet
	 * sich nach der Zeichenreihenfolge der Komponenten: Je weiter eine
	 * Komponente im Vordergrund ist, desto frher wird sie benachrichtigt. Die
	 * Bhne wird nach allen Komponenten benachrichtigt. Durch Konsumieren des
	 * Events, kann die Benachrichtigungssequenz abgebrochen werden.
	 * 
	 * @param e
	 *            enthlt Details zum eingetretenen Event
	 */
	public void mouseExited(MouseInfo e) {
	}

}
