package theater_intern;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import simulation.SimulationManager;
import theater.Actor;
import theater.Component;
import theater.PixelArea;
import theater.Stage;
import theater.TheaterImage;
import util.Utils;

/**
 * Interne Umsetzung einer Bhne
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class IStage {

	private static HashMap<Stage, IStage> stageMap = new HashMap<Stage, IStage>();
	private static HashMap<IStage, Stage> iStageMap = new HashMap<IStage, Stage>();
	
	public static void reinit() {
		stageMap = new HashMap<Stage, IStage>();
		iStageMap = new HashMap<IStage, Stage>();
	}

	public static void newStageIStage(Stage stage, IStage iStage) {
		IStage.stageMap.put(stage, iStage);
		IStage.iStageMap.put(iStage, stage);
	}

	public static Stage getStage(IStage iStage) {
		return IStage.iStageMap.get(iStage);
	}

	public static IStage getIStage(Stage stage) {
		return IStage.stageMap.get(stage);
	}

	private ArrayList<IComponent> allComponents;
	private TheaterImage background;
	private int cellSize;
	private int noOfCols;
	private int noOfRows;
	private boolean tiled;
	private boolean handlingKeyEvents;
	private boolean handlingMouseEvents;

	public IStage(int noOfCols, int noOfRows, int cellSize) {
		this.noOfCols = noOfCols;
		this.noOfRows = noOfRows;
		this.cellSize = cellSize;
		this.tiled = true;
		this.allComponents = new ArrayList<IComponent>();
		this.background = null; // new TheaterImage(noOfCols, noOfRows);
		this.handlingKeyEvents = true;
		this.handlingMouseEvents = true;
	}

	public static void setStage(IStage stage) {
		SimulationManager.getSimulationManager().setStage(stage);
	}

	public void add(IComponent comp, int col, int row) {
		if (this.allComponents.contains(comp) || comp.getStage() != null) {
			return;
		}
		this.allComponents.add(comp);
		comp.setStage(this);
		Component component = IComponent.getComp(comp);
		component.setLocation(col, row);
		component.addedToStage(IStage.getStage(this));
		TheaterObservable.getObservable().stateChange();
	}

	public void remove(IComponent comp) {
		rem(comp);
		TheaterObservable.getObservable().stateChange();
	}

	public void remove(List<Component> comps) {
		for (Component comp : comps) {
			IComponent icomp = IComponent.getIComp(comp);
			rem(icomp);
		}
		TheaterObservable.getObservable().stateChange();
	}

	private void rem(IComponent comp) {
		if (!this.allComponents.contains(comp)) {
			return;
		}
		this.allComponents.remove(comp);
		comp.setStage(null);
	}

	public int getNumberOfColumns() {
		return this.noOfCols;
	}

	public int getNumberOfRows() {
		return this.noOfRows;
	}

	public int getCellSize() {
		return this.cellSize;
	}

	public void setSize(int noOfCols, int noOfRows) {
		this.noOfCols = noOfCols;
		this.noOfRows = noOfRows;
		TheaterObservable.getObservable().stateChange();
	}

	public void setSize(int noOfCols, int noOfRows, int cellSize) {
		this.noOfCols = noOfCols;
		this.noOfRows = noOfRows;
		this.cellSize = cellSize;
		TheaterObservable.getObservable().stateChange();
	}

	public boolean isTiled() {
		return this.tiled;
	}

	public final void setBackground(TheaterImage image) {
		if (this.background != image) {
			this.background = image;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public final void setBackground(String filename)
			throws IllegalArgumentException {
		this.setBackground(new TheaterImage(filename));
	}

	public TheaterImage getBackground() {
		return this.background;
	}

	public void paint() {
		TheaterObservable.getObservable().importantStateChange();
	}

	public List<IComponent> getComponentsInPaintOrder() {
		Collections.sort(this.allComponents, new IComponentComparator());
		return this.allComponents;
	}

	public List<Actor> getActorsInActOrder() {

		List<Actor> actors = new ArrayList<Actor>();
		for (IComponent icomp : this.allComponents) {
			Component comp = IComponent.getComp(icomp);
			if (Utils.dynInstanceof(comp, Actor.class)) {
				actors.add((Actor) comp);
			}
		}
		Collections.sort(actors, new ActorComparator());
		return actors;

	}

	public int numberOfComponents() {
		return this.allComponents.size();
	}

	public List<IComponent> getAllIComponents() {
		return new ArrayList<IComponent>(this.allComponents);
	}

	static java.awt.Point toAWTPoint(theater.Point p) {
		return new java.awt.Point(p.getX(), p.getY());
	}

	public List<Component> getComponents(Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>();
		for (IComponent iComp : this.allComponents) {
			Component comp = IComponent.getComp(iComp);
			if (IStage.isInstanceOf(comp, classes)) {
				comps.add(comp);
			}
		}
		return comps;
	}

	public List<Component> getComponentsAt(int col, int row,
			Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>();
		for (IComponent iComp : this.allComponents) {
			if (iComp.getRow() == row && iComp.getColumn() == col) {
				Component comp = IComponent.getComp(iComp);
				if (IStage.isInstanceOf(comp, classes)) {
					comps.add(comp);
				}
			}
		}
		return comps;
	}

	public List<Component> getComponentsInside(PixelArea area,
			Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>();
		for (IComponent iComp : this.allComponents) {
			Component comp = IComponent.getComp(iComp);
			if (IStage.isInstanceOf(comp, classes) && iComp.isInside(area)) {
				comps.add(comp);
			}
		}
		return comps;
	}

	public List<Component> getIntersectingComponents(PixelArea area,
			Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>();
		for (IComponent iComp : this.allComponents) {
			Component comp = IComponent.getComp(iComp);
			if (IStage.isInstanceOf(comp, classes) && iComp.intersects(area)) {
				comps.add(comp);
			}
		}
		return comps;
	}

	static boolean isInstanceOf(Component comp, Class<?>... classes) {
		if (classes == null || classes.length == 0) {
			return true;
		}
		for (Class<?> c : classes) {
			if (Utils.dynInstanceof(comp, c)) {
				return true;
			}
		}
		return false;
	}

	public boolean isHandlingKeyEvents() {
		return this.handlingKeyEvents;
	}

	public void setHandlingKeyEvents(boolean handlingKeyEvents) {
		this.handlingKeyEvents = handlingKeyEvents;
	}

	public boolean isHandlingMouseEvents() {
		return this.handlingMouseEvents;
	}

	public void setHandlingMouseEvents(boolean handlingMouseEvents) {
		this.handlingMouseEvents = handlingMouseEvents;
	}

}
