package theater;

import java.awt.Color;
import java.awt.Container;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Toolkit;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.net.URL;

import javax.imageio.ImageIO;

import listener.Application;
import model.Play;
import theater_intern.TheaterObservable;
import util.ResourceLoader;

/**
 * TheaterImage ist eine Theater-Klasse mit vielfltigen Methoden zum Erzeugen
 * und Manipulieren von Bildern bzw. Ikons, die dann Akteuren, Requisiten oder
 * der Bhne zugeordnet werden knnen. Die Bilder lassen sich dabei auch noch
 * zur Laufzeit verndern, so dass mit Hilfe der Klasse TheaterImage bspw.
 * Punktezhler fr kleinere Spiele implementiert werden knnen.
 * 
 * @author Dietrich Boles, Universitt Oldenburg, Germany
 * @version 1.0 (08.09.2009)
 * 
 */
public class TheaterImage {

	/**
	 * intern verwendetes Objekt
	 */
	private Graphics2D g;

	/**
	 * intern verwendetes Objekt
	 */
	private BufferedImage image;

	/**
	 * Transparenz
	 */
	protected int transparency = 255;

	/**
	 * Konstruktor zur Initialisierung eines TheaterImages mit dem Default-Image
	 * der entsprechenden Klasse:
	 * <ul>
	 * <li>Stage: Bhne mit Vorhang</li>
	 * <li>Actor: Marionette</li>
	 * <li>Prop: Sessel</li>
	 * <li>Performance: Flagge</li>
	 * <li>ansonsten: Werkzeuge</li>
	 * </ul>
	 * 
	 * Die Bilder werden im Dateibereich des Theater-Simulators benutzt. Die
	 * Default-Images werden auch benutzt, wenn eine Actor- bzw. Prop-Klasse
	 * ihren Objekten keine eigenen Ikons zuordnet.
	 * 
	 * @param father
	 *            Klassenobjekt der entsprechenden Klasse (Stage.class,
	 *            Actor.class, Prop.class, Performance.class)
	 */
	public TheaterImage(Class<?> father) {
		Image img = null;
		if (father == Actor.class) {
			img = ResourceLoader.getImage("actor.gif");
		} else if (father == Prop.class) {
			img = ResourceLoader.getImage("prop.gif");
		} else if (father == Stage.class) {
			img = ResourceLoader.getImage("stage.gif");
		} else if (father == Performance.class) {
			img = ResourceLoader.getImage("performance.gif");
		} else { // other
			img = ResourceLoader.getImage("other.gif");
		}

		try {
			this.image = new BufferedImage(img.getWidth(null), img
					.getHeight(null), BufferedImage.TYPE_INT_ARGB);
			Graphics2D gr = (Graphics2D) this.image.getGraphics();
			gr.drawImage(img, 0, 0, null);
			gr.dispose();
			this.init();
		} catch (Throwable exc) {
			exc.printStackTrace();
			throw new IllegalArgumentException();
		}
	}

	/**
	 * Konstruktor zum Initialisieren eines TheaterImage mit einem Bild aus
	 * einer Datei. Erlaubte Bildformate sind gif, jpq und png.
	 * 
	 * @param filename
	 *            Name der Bilddatei; die Datei muss sich im Unterverzeichnis
	 *            "images" des Theaterstcks befinden
	 * @throws IllegalArgumentException
	 *             wird geworfen, wenn die Datei nicht existiert, keine gltige
	 *             Bilddatei ist oder nicht lesbar ist
	 */
	public TheaterImage(String filename) throws IllegalArgumentException {
		String dir = Play.getPlay().getDirectory();
		if (Application.isApplet()) {
			try {
				MediaTracker tracker = new MediaTracker(new Container());
				URL url = TheaterImage.class.getClassLoader().getResource(
						"data/images/" + filename);
				Image img = Toolkit.getDefaultToolkit().createImage(url);
				tracker.addImage(img, 1);

				tracker.waitForID(1);
				this.image = new BufferedImage(img.getWidth(null), img
						.getHeight(null), BufferedImage.TYPE_INT_ARGB);
				Graphics2D gr = (Graphics2D) this.image.getGraphics();
				gr.drawImage(img, 0, 0, null);
				gr.dispose();
				this.init();
			} catch (Throwable exc) {
				exc.printStackTrace();
				throw new IllegalArgumentException();
			}
		} else if (Application.isStandAlone()) {
			try {
				this.image = ImageIO.read(ResourceLoader
						.getDataFileURL("data/images/" + filename));
				this.init();
			} catch (IOException exc) {
				throw new IllegalArgumentException();
			}
		} else {
			File f = new File(dir + File.separatorChar + "images"
					+ File.separatorChar + filename);
			if (!f.exists()) {
				throw new IllegalArgumentException();
			}
			try {
				this.image = ImageIO.read(f);
				this.init();
			} catch (IOException exc) {
				throw new IllegalArgumentException();
			}
		}

	}

	/**
	 * Konstruktor zum Erzeugen eines leeren TheaterImages in einer bestimmten
	 * Gre
	 * 
	 * @param width
	 *            Breite des Bildes in Pixeln (> 0)
	 * @param height
	 *            Hhe des Bildes in Pixeln (> 0)
	 */
	public TheaterImage(int width, int height) {
		this.image = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_ARGB);
		this.init();
	}

	/**
	 * Copykonstruktor zum Initialisieren eines TheaterImages mit einem bereits
	 * existierenden TheaterImage
	 * 
	 * @param im
	 *            ein bereits existierendes TheaterImage (darf nicht null sein)
	 */
	public TheaterImage(TheaterImage im) {
		this.image = new BufferedImage(im.getWidth(), im.getHeight(),
				BufferedImage.TYPE_INT_ARGB);
		this.init();
		this.transparency = im.transparency;
		this.g.drawImage(im.image, 0, 0, Play.getPlay().getStagePanel());
	}

	/**
	 * interner Default-Konstruktor
	 */
	protected TheaterImage() {
		this.g = null;
	}

	/**
	 * Liefert die Breite des TheaterImages in Pixeln.
	 * 
	 * @return die Breite des TheaterImages in Pixeln
	 */
	public int getWidth() {
		return this.image.getWidth();
	}

	/**
	 * Liefert die Hhe des TheaterImages in Pixeln.
	 * 
	 * @return die Hhe des TheaterImages in Pixeln
	 */
	public int getHeight() {
		return this.image.getHeight();
	}

	/**
	 * Setzt die Transparenz; Standardmaessig ist der Transparenzwert 255 (gar
	 * nicht durchsichtig)
	 * 
	 * @param t
	 *            Wert zwischen 0 (ganz durchsichtig) und 255 (gar nicht
	 *            durchsichtig)
	 */
	public void setTransparency(int t) {
		this.transparency = t;
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Liefert die Transparenz
	 * 
	 * @return die eingestellte Transparenz; Wert zwischen 0 (ganz durchsichtig)
	 *         und 255 (gar nicht durchsichtig)
	 */
	public int getTransparency() {
		return this.transparency;
	}

	/**
	 * Ordnet dem TheaterImage eine Farbe zu, in der bei Aufruf der draw- bzw.
	 * fill-Methoden die entsprechenden Graphik-Primitiven gezeichnet werden.
	 * 
	 * @param color
	 *            die neue Zeichenfarbe
	 */
	public void setColor(java.awt.Color color) {
		this.g.setColor(color);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Liefert die aktuelle Zeichenfarbe des TheaterImages.
	 * 
	 * @return die aktuelle Zeichenfarbe des TheaterImages
	 */
	public java.awt.Color getColor() {
		return this.g.getColor();
	}

	/**
	 * Setzt den Font, in dem Texte durch nachfolgende Aufrufe der
	 * drawString-Methode in dem TheaterImage gezeichnet werden sollen.
	 * 
	 * @param f
	 *            der neue Font
	 */
	public void setFont(java.awt.Font f) {
		this.g.setFont(f);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Liefert den aktuellen Font des TheaterImages.
	 * 
	 * @return der aktuelle Font des TheaterImages
	 */
	public java.awt.Font getFont() {
		return this.g.getFont();
	}

	/**
	 * Zeichnet im TheaterImage eine Linie in der aktuellen Zeichenfarbe.
	 * 
	 * @param x1
	 *            x-Koordinate, von der aus die Linie gezeichnet werden soll
	 * @param y1
	 *            y-Koordinate, von der aus die Linie gezeichnet werden soll
	 * @param x2
	 *            x-Koordinate, bis wohin die Linie gezeichnet werden soll
	 * @param y2
	 *            y-Koordinate, bis wohin die Linie gezeichnet werden soll
	 */
	public void drawLine(int x1, int y1, int x2, int y2) {
		this.g.drawLine(x1, y1, x2, y2);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage ein Rechteck in der aktuellen Zeichenfarbe.
	 * 
	 * @param x
	 *            x-Koordinate der linken oberen Ecke des Rechtecks
	 * @param y
	 *            y-Koordinate der linken oberen Ecke des Rechtecks
	 * @param width
	 *            Breite des Rechtecks (in Pixeln)
	 * @param height
	 *            Hhe des Rechtecks (in Pixeln)
	 */
	public void drawRect(int x, int y, int width, int height) {
		this.g.drawRect(x, y, width, height);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage ein Oval in der aktuellen Zeichenfarbe.
	 * 
	 * @param x
	 *            x-Koordinate der linken oberen Ecke des Ovals
	 * @param y
	 *            y-Koordinate der linken oberen Ecke des Ovals
	 * @param width
	 *            Breite des Ovals in Pixeln
	 * @param height
	 *            Hhe des Ovals in Pixeln
	 */
	public void drawOval(int x, int y, int width, int height) {
		this.g.drawOval(x, y, width, height);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage ein Polygon in der aktuellen Zeichenfarbe. Es
	 * wird automatisch ein Linie hinzugefgt, die das Polygon schliet.
	 * 
	 * @param xPoints
	 *            x-Koordinaten der Linien
	 * @param yPoints
	 *            y-Koordinaten der Linien
	 * @param nPoints
	 *            Anzahl der Liniensegmente
	 */
	public void drawPolygon(int[] xPoints, int[] yPoints, int nPoints) {
		this.g.drawPolygon(xPoints, yPoints, nPoints);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage einen Text im aktuell gesetzten Font.
	 * 
	 * @param string
	 *            der zu zeichnende Text
	 * @param x
	 *            x-Koordinate, an der der Text beginnen soll
	 * @param y
	 *            y-Koordinate, an der der Text beginnen soll
	 */
	public void drawString(String string, int x, int y) {
		this.g.drawString(string, x, y);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet ein existierendes TheaterImage an einer bestimmten Stelle in das
	 * aufgerufene TheaterImage
	 * 
	 * @param image
	 *            das TheaterImage, das gezeichnet werden soll (darf nicht null
	 *            sein)
	 * @param x
	 *            x-Koordinate, an der das Image gezeichnet werden soll
	 * @param y
	 *            y-Koordinate, an der das Image gezeichnet werden soll
	 */
	public void drawImage(TheaterImage image, int x, int y) {
		this.g.drawImage(image.image, x, y, null);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Fllt das gesamte TheaterImage in der aktuellen Zeichenfarbe.
	 */
	public void fill() {
		this.g.fillRect(0, 0, this.image.getWidth(), this.image.getHeight());
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage ein geflltes Rechteck in der aktuellen
	 * Zeichenfarbe.
	 * 
	 * @param x
	 *            x-Koordinate der linken oberen Ecke des Rechtecks
	 * @param y
	 *            y-Koordinate der linken oberen Ecke des Rechtecks
	 * @param width
	 *            Breite des Rechtecks (in Pixeln)
	 * @param height
	 *            Hhe des Rechtecks (in Pixeln)
	 */
	public void fillRect(int x, int y, int width, int height) {
		this.g.fillRect(x, y, width, height);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage ein geflltes Oval in der aktuellen
	 * Zeichenfarbe.
	 * 
	 * @param x
	 *            x-Koordinate der linken oberen Ecke des Ovals
	 * @param y
	 *            y-Koordinate der linken oberen Ecke des Ovals
	 * @param width
	 *            Breite des Ovals in Pixeln
	 * @param height
	 *            Hhe des Ovals in Pixeln
	 */
	public void fillOval(int x, int y, int width, int height) {
		this.g.fillOval(x, y, width, height);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Zeichnet im TheaterImage ein geflltes Polygon in der aktuellen
	 * Zeichenfarbe. Es wird automatisch ein Linie hinzugefgt, die das Polygon
	 * schliet.
	 * 
	 * @param xPoints
	 *            x-Koordinaten der Linien
	 * @param yPoints
	 *            y-Koordinaten der Linien
	 * @param nPoints
	 *            Anzahl der Liniensegmente
	 */
	public void fillPolygon(int[] xPoints, int[] yPoints, int nPoints) {
		this.g.fillPolygon(xPoints, yPoints, nPoints);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Lscht ein TheaterImage.
	 */
	public void clear() {
		Color c = this.g.getColor();
		Color bc = this.g.getBackground();
		this.g.setBackground(new Color(255, 255, 255, 0));
		this.g.clearRect(0, 0, this.image.getWidth(), this.image.getHeight());
		this.g.setBackground(bc);
		this.g.setColor(c);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Setzt ein bestimmtes Pixel des TheaterImages auf eine bestimmte Farbe.
	 * 
	 * @param x
	 *            x-Koordinate des Pixels
	 * @param y
	 *            y-Koordinate des Pixels
	 * @param color
	 *            neue Farbe des Pixels
	 */
	public void setColorAt(int x, int y, java.awt.Color color) {
		int argb = color.getRGB();
		this.image.setRGB(x, y, argb);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Liefert die Farbe eines bestimmten Pixels des TheaterImages.
	 * 
	 * @param x
	 *            x-Koordinate des Pixels
	 * @param y
	 *            y-Koordinate des Pixels
	 * @return die Farbe eines bestimmten Pixels des TheaterImages
	 */
	public java.awt.Color getColorAt(int x, int y) {
		int argb = this.image.getRGB(x, y);
		int alpha = argb >> 24 & 0xff;
		int red = argb >> 16 & 0xff;
		int green = argb >> 8 & 0xff;
		int blue = argb & 0xff;
		return new Color(red, green, blue, alpha);
	}

	/**
	 * Spiegelt das TheaterImage horizontal. Achtung: Die Gre des Bildes wird
	 * dabei nicht verndert!
	 */
	public void mirrorHorizontally() {
		Color oldC = this.getColor();
		Font oldF = this.getFont();
		AffineTransform transformObj = AffineTransform.getTranslateInstance(
				this.image.getWidth(), 0);
		transformObj.scale(-1.0, 1.0);
		AffineTransformOp filterObj = new AffineTransformOp(transformObj,
				AffineTransformOp.TYPE_BICUBIC);

		BufferedImage dest = filterObj.createCompatibleDestImage(this.image,
				null);// this.image.getColorModel());

		filterObj.filter(this.image, dest);
		this.image = dest;
		if (this.g != null) {
			this.g.dispose();
		}
		this.g = this.image.createGraphics();
		this.g.setColor(oldC);
		this.g.setFont(oldF);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Spiegelt das TheaterImage vertikal. Achtung: Die Gre des Bildes wird
	 * dabei nicht verndert!
	 */
	public void mirrorVertically() {
		Color oldC = this.getColor();
		Font oldF = this.getFont();
		AffineTransform transformObj = AffineTransform.getTranslateInstance(0,
				this.image.getHeight());
		transformObj.scale(1.0, -1.0);
		AffineTransformOp filterObj = new AffineTransformOp(transformObj,
				AffineTransformOp.TYPE_BICUBIC);

		BufferedImage dest = filterObj.createCompatibleDestImage(this.image,
				null); // this.image.getColorModel());

		filterObj.filter(this.image, dest);
		this.image = dest;
		if (this.g != null) {
			this.g.dispose();
		}
		this.g = this.image.createGraphics();
		this.g.setColor(oldC);
		this.g.setFont(oldF);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Dreht das TheaterImage um eine bestimmte Gradzahl. Achtung: Die Gre des
	 * Bildes wird dabei nicht verndert!
	 * 
	 * @param degrees
	 *            Gradzahl der Drehung
	 */
	public void rotate(int degrees) {
		Color oldC = this.getColor();
		Font oldF = this.getFont();
		AffineTransform affineTransform = AffineTransform.getRotateInstance(
				Math.toRadians(degrees), this.image.getWidth() / 2, this.image
						.getHeight() / 2);
		BufferedImage rotatedImage = new BufferedImage(this.image.getWidth(),
				this.image.getHeight(), BufferedImage.TYPE_INT_ARGB);
		Graphics2D gr = (Graphics2D) rotatedImage.getGraphics();
		gr.setTransform(affineTransform);
		gr.drawImage(this.image, 0, 0, null);
		this.image = rotatedImage;
		if (this.g != null) {
			this.g.dispose();
		}
		this.g = this.image.createGraphics();
		this.g.setColor(oldC);
		this.g.setFont(oldF);
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Skaliert das TheaterImage auf einebestimmte Gre.
	 * 
	 * @param width
	 *            die neue Breite des TheaterImages
	 * @param height
	 *            die neue Hhe des TheaterImages
	 */
	public void scale(int width, int height) {
		BufferedImage bim = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_ARGB);
		bim.getGraphics().drawImage(this.image, 0, 0, width, height, null);
		this.image = bim;
		if (this.g != null) {
			this.g.dispose();
		}
		this.g = this.image.createGraphics();
		TheaterObservable.getObservable().stateChange();
	}

	/**
	 * Intern wird ein TheaterImage durch ein
	 * java.awt.image.BufferedImage-Objekt realisiert. Diese Methode liefert das
	 * entsprechende Objekt. Achtung: Einige Methoden tauschen intern das
	 * BufferedImage-Objekt aus!
	 * 
	 * @return das aktuelle interne BufferedImage-Objekt
	 */
	public java.awt.Image getAwtImage() {
		return this.image;
	}

	/**
	 * interne init-Methode
	 */
	private void init() {
		if (this.g != null) {
			this.g.dispose();
		}
		this.g = this.image.createGraphics();
		this.g.setColor(Color.BLACK);
	}

}
